<?php

namespace Drupal\kinetic;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Layout\LayoutDefault;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Render\Element;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Kinetic Base Layout class. Add all section configuration here!
 *
 * @internal
 *   Plugin classes are internal.
 */
class KineticLayout extends LayoutDefault implements ContainerFactoryPluginInterface {
  /**
   * Entity Type Manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'bg_color' => 'white',
      'content_width' => 'container-default',
      'spacing' => 'default',
      'v_align' => 'top',
      'mobile_order' => 'stacked',
      'section_widths' => '50-50',
      'label' => '',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $configuration = $this->getConfiguration();

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Layout Title'),
      '#description' => $this->t('This label will be used as a title for this layout in the Layout Builder UI.'),
      '#default_value' => $configuration['label'],
    ];

    $form['bg_color'] = [
      '#type' => 'select',
      '#title' => $this->t('Background Color'),
      '#options' => [
        'bg-white' => 'White',
        'bg-gray-100' => 'Gray',
        'bg-blue-100' => 'Light Blue',
        'bg-blue' => 'Blue',
        'bg-blue-500' => 'Dark Blue',
      ],
      '#default_value' => $configuration['bg_color'],
    ];

    $form['content_width'] = [
      '#type' => 'select',
      '#title' => $this->t('Content Width'),
      '#options' => [
        'container-default' => 'Default',
        'container-full' => 'Full Width',
        'container-narrow' => 'Narrow',
        'container-wide' => 'Wide',
      ],
      '#default_value' => $configuration['content_width'],
    ];

    $form['spacing'] = [
      '#type' => 'select',
      '#title' => t('Section spacing'),
      '#default_value' => $configuration['spacing'],
      '#description' => t('Control spacing on the top and bottom of the section.'),
      '#options' => [
        'default' => 'Default',
        'no_top' => 'No Top Spacing',
        'no_bottom' => 'No Bottom Spacing',
        'no_spacing' => 'No Spacing',
      ],
    ];

    $form['section_widths'] = [
      '#type' => 'select',
      '#title' => $this->t('Section Widths'),
      '#options' => [
        '50-50' => '50% | 50%',
        '33-66' => '33% | 66%',
        '66-33' => '66% | 33%',
      ],
      '#default_value' => $configuration['section_widths'],
    ];

    $form['v_align'] = [
      '#type' => 'select',
      '#title' => $this->t('Vertical Alignment'),
      '#description' => $this->t('Vertical alignment of the regions inside the section.'),
      '#options' => [
        'top' => 'Top',
        'center' => 'Center',
        'bottom' => 'Bottom',
      ],
      '#default_value' => $configuration['v_align'],
    ];

    $form['mobile_order'] = [
      '#type' => 'select',
      '#title' => $this->t('Mobile Order'),
      '#options' => [
        'stacked' => 'Stacked',
        'two-col' => 'Two Columns',
      ],
      '#default_value' => $configuration['mobile_order'],
    ];

    // Disable all the options by default. Classes that extend this form should enable the options it needs.
    foreach(Element::children($form) as $item) {
      $form[$item]['#access'] = FALSE;
    }

    $form['label']['#access'] = TRUE;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    // If the form item is disabled, set config value to NULL.
    foreach(Element::children($form) as $item) {
      if ($form_state->hasValue($item)) {
        $is_disabled = isset($form[$item]['#access']) && !$form[$item]['#access'];
        $this->configuration[$item] = $is_disabled ? NULL : $form_state->getValue($item);
      }
    }
  }

}
