import { defineConfig } from "vite";
import fg from 'glob';
import { resolve } from "path";
import SdcLibraryGenerator from "./build-utils/SdcLibraryGenerator";
import tailwindcss from '@tailwindcss/vite';
import postcssNested from 'postcss-nested';
import autoprefixer from 'autoprefixer';

const port = 5173;
// Assumes you are using DDEV.
// These are undefined outside of that environment.
const { HTTPS_EXPOSE, DDEV_TLD, DDEV_SITENAME } = process.env;
const protocol = HTTPS_EXPOSE && HTTPS_EXPOSE.includes(port) ? 'https' : 'http';
const origin = `${protocol}://${DDEV_SITENAME}.${DDEV_TLD}:${port}`;

let files = new Map(
  fg.sync('components/**/*.{pcss.css,es6.js}').map(file => [
    file, file,
  ]),
);

files.set('global', './source/01-base/global/css/index.pcss.css');
files.set('layout-builder', './source/01-base/global/css/layout-builder.pcss.css');
files.set('wysiwyg', './source/01-base/global/css/wysiwyg.pcss.css');
files.set('global-js', './source/01-base/global/js/index.es6.js');

export default defineConfig({
  plugins: [
    tailwindcss({
      base: './source/01-base/global/css',
    }),
    SdcLibraryGenerator({
      baseDir: './source/02-components',
    }),
  ],
  resolve: {
    extensions: ['.js'],
    alias: {
      '@base': resolve(__dirname, './source/01-base'),
    },
  },
  build: {
    manifest: true,
    rollupOptions: {
      input: Object.fromEntries(files),
      output: {
        assetFileNames: (assetInfo) => {
          // dont hash the ckeditor css
          if (assetInfo.names.includes('wysiwyg.css')) {
            return `assets/wysiwyg.css`;
          }
          if (assetInfo.names.includes('global.css')) {
            return `assets/global.css`;
          }
          return `assets/[name]-[hash].[ext]`;
        },
      },
    },
  },
  css: {
    devSourcemap: true,
    postcss: {
      plugins: [
        postcssNested(),
        autoprefixer(),
      ],
    },
  },
  server: {
    host: "0.0.0.0",
    port: port,
    strictPort: true,
    origin: origin,
    cors: {
      origin: /https?:\/\/([A-Za-z0-9\-\.]+)?(\.ddev\.site)(?::\d+)?$/,
    },
  },
});
