/**
 * Knova Widget JavaScript
 */
(function($) {
	'use strict';

	// Generate unique session ID
	function generateSessionId() {
		return 'knova_' + Date.now() + '_' + Math.random().toString(36).substring(2, 11);
	}

	// Get or create session ID
	function getSessionId() {
		let sessionId = localStorage.getItem('knova_session_id');
		if (!sessionId) {
			sessionId = generateSessionId();
			localStorage.setItem('knova_session_id', sessionId);
		}
		return sessionId;
	}

	// Format markdown to HTML
	function formatMarkdown(text) {
		if (!text) return '';

		// Headers
		text = text.replace(/^#{1}\s+(.*)/gm, '<h1>$1</h1>');
		text = text.replace(/^#{2}\s+(.*)/gm, '<h2>$1</h2>');
		text = text.replace(/^#{3}\s+(.*)/gm, '<h3>$1</h3>');

		// Bold
		text = text.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
		text = text.replace(/__(.*?)__/g, '<strong>$1</strong>');

		// Italic
		text = text.replace(/\*(.*?)\*/g, '<em>$1</em>');
		text = text.replace(/_(.*?)_/g, '<em>$1</em>');

		// Links
		text = text.replace(/\[([^\]]*)\]\(([^\)]*)\)/g, '<a href="$2" target="_blank" rel="noopener">$1</a>');

		// Lists
		text = text.replace(/^\d+\.\s+(.*)$/gm, '<li>$1</li>');
		text = text.replace(/^[-*]\s+(.*)$/gm, '<li>$1</li>');
		
		// Wrap consecutive list items in ul
		text = text.replace(/(<li>.*<\/li>\n?)+/g, function(match) {
			return '<ul>' + match.replace(/\n/g, '') + '</ul>';
		});

		// Paragraphs
		text = text.split('\n\n').map(function(paragraph) {
			paragraph = paragraph.trim();
			if (paragraph && !paragraph.match(/^<[h|u|o|l]/)) {
				return '<p>' + paragraph + '</p>';
			}
			return paragraph;
		}).join('');

		// Line breaks
		text = text.replace(/\n/g, '<br>');

		return text;
	}

	// Typewriter effect with auto-scroll
	function typewriter(element, html, callback) {
		const text = formatMarkdown(html);
		let i = 0;
		let output = '';
		const chars = text.split('');
		
		// Get the messages container for scrolling
		const messagesContainer = $('#knova-messages-container');
		const scrollContainer = messagesContainer.length ? messagesContainer.parent() : null;

		function addChar() {
			if (i < chars.length) {
				// Handle HTML tags
				if (chars[i] === '<') {
					let tag = '';
					while (i < chars.length && chars[i] !== '>') {
						tag += chars[i];
						i++;
					}
					if (i < chars.length) {
						tag += chars[i];
						i++;
					}
					output += tag;
					element.html(output);
					// Scroll during typing (every few characters for performance)
					if (i % 5 === 0 && scrollContainer) {
						forceScrollToBottom(scrollContainer);
					}
					addChar();
				} else {
					output += chars[i];
					element.html(output);
					i++;
					// Scroll during typing (every few characters for performance)
					if (i % 5 === 0 && scrollContainer) {
						forceScrollToBottom(scrollContainer);
					}
					setTimeout(addChar, 10);
				}
			} else {
				// Final scroll when typing is complete
				if (scrollContainer) {
					forceScrollToBottom(scrollContainer);
				}
				if (callback) callback();
			}
		}

		addChar();
	}

	// Add message to chat
	function addMessage(container, message, role) {
		const messageDiv = $('<div>').addClass('knova-message knova-message-' + role);
		const bubble = $('<div>').addClass('knova-message-bubble');
		
		if (role === 'user') {
			bubble.text(message);
		} else {
			bubble.html(formatMarkdown(message));
		}
		
		messageDiv.append(bubble);
		container.append(messageDiv);
		
		// Check if assistant message suggests getting in touch
		if (role === 'assistant') {
			const text = message.toLowerCase();
			const contactKeywords = ['contact', 'get in touch', 'reach out', 'call us', 'email us', 'book', 'appointment', 'schedule', 'provide your', 'share your contact', 'contact information', 'email address', 'phone number'];
			if (contactKeywords.some(keyword => text.includes(keyword))) {
				// Wait a bit for the message to be fully rendered
				setTimeout(function() {
					// Only add button if it doesn't already exist after this message
					if (!messageDiv.next('.knova-show-inquiry-btn').length) {
						const showInquiryBtn = $('<button type="button" class="knova-show-inquiry-btn">Get in Touch</button>');
						showInquiryBtn.on('click', function() {
							showInquiryForm();
						});
						messageDiv.after(showInquiryBtn);
					}
				}, 200);
			}
		}
		
		// Scroll to bottom - use parent container for proper scrolling
		const scrollContainer = container.parent();
		if (scrollContainer.length) {
			scrollToBottom(scrollContainer);
		} else {
			scrollToBottom(container);
		}
	}

	// Scroll to bottom with smooth animation
	function scrollToBottom(container) {
		if (!container || !container.length) {
			return;
		}
		const scrollContainer = container[0];
		if (scrollContainer) {
			scrollContainer.scrollTo({
				top: scrollContainer.scrollHeight,
				behavior: 'smooth'
			});
		}
	}
	
	// Force scroll to bottom (instant, for immediate updates)
	function forceScrollToBottom(container) {
		if (!container || !container.length) {
			return;
		}
		const scrollContainer = container[0];
		if (scrollContainer) {
			scrollContainer.scrollTop = scrollContainer.scrollHeight;
		}
	}

	// Show/hide typing indicator
	function toggleTypingIndicator(show) {
		const typingIndicator = $('#knova-typing-indicator');
		if (show) {
			typingIndicator.show();
			scrollToBottom($('#knova-messages-container').parent());
		} else {
			typingIndicator.hide();
		}
	}

	// Initialize widget
	function initWidget() {
		const container = $('#knova-widget-container');
		if (!container.length) {
			console.warn('Knova: Widget container not found in DOM');
			return;
		}
		
		// Mark as initialized to prevent multiple initializations
		if (container.data('knova-initialized')) {
			return;
		}
		container.data('knova-initialized', true);
		

		const widget = $('#knova-widget');
		const toggle = $('#knova-widget-toggle');
		
		// Verify critical elements exist
		if (!widget.length) {
			console.error('Knova: Widget element not found');
			return;
		}
		if (!toggle.length) {
			console.error('Knova: Toggle button not found');
			return;
		}
		const closeBtn = $('#knova-close-button');
		const backBtn = $('#knova-back-button');
		const menuBtn = $('#knova-menu-button');
		const menuDropdown = $('#knova-menu-dropdown');
		const expandMenuItem = $('#knova-expand-menu-item');
		const collapseMenuItem = $('#knova-collapse-menu-item');
		const clearChatItem = $('#knova-clear-chat-item');
		const form = $('#knova-input-form');
		const input = $('#knova-input-field');
		const sendBtn = $('#knova-send-button');
		const messagesContainer = $('#knova-messages-container');

		let isOpen = false;
		let isExpanded = false;
		let isMenuOpen = false;
		let sessionId = getSessionId();

		// Check if elements exist
		if (!menuBtn.length || !menuDropdown.length || !expandMenuItem.length || !collapseMenuItem.length) {
			console.error('Knova: Menu elements not found');
		}
		
		if (!closeBtn.length) {
			console.error('Knova: Close button not found');
		}

		// Initialize menu state
		menuDropdown.hide();
		collapseMenuItem.hide();

		// Get settings from Drupal
		const settings = (typeof drupalSettings !== 'undefined' && drupalSettings.knova) ? drupalSettings.knova : {};
		
		// Get default dimensions from settings (these come from admin form)
		const defaultHeight = settings.widgetHeightMax || 400; // From admin settings
		const defaultWidth = settings.widgetWidthDesktop || 380; // From admin settings
		const expandedWidth = 600; // Expanded width

		// The height is set by inline CSS from PHP (with !important)
		// We should use the value from settings, not override it with inline styles
		// Only set width if not already set by CSS
		if (!widget.css('width') || widget.css('width') === 'auto') {
			widget.css('width', defaultWidth + 'px');
		}
		
		// Store the target height for animations (from settings, which matches the CSS)
		const targetHeight = defaultHeight;

		// Toggle widget with smooth animation
		function toggleWidget() {
			if (isOpen) {
				// Smooth close animation - fade out and slide down
				widget.animate({
					opacity: 0,
					height: 0
				}, {
					duration: 300,
					easing: 'swing',
					complete: function() {
						widget.hide().css({
							opacity: '',
							height: '' // Remove inline height to let CSS handle it
						});
						// Reset to collapsed state when closing
						if (isExpanded) {
							collapseWidget();
						}
						closeMenu();
					}
				});
				isOpen = false;
				backBtn.hide();
			} else {
				// Smooth open animation - fade in and slide up
				const currentWidth = isExpanded ? expandedWidth : defaultWidth;
				
				widget.css({
					display: 'flex',
					opacity: 0,
					height: 0,
					width: currentWidth + 'px'
				}).show();
				
				widget.animate({
					opacity: 1,
					height: targetHeight + 'px'
				}, {
					duration: 300,
					easing: 'swing',
					complete: function() {
						// Ensure correct width when opening
						if (isExpanded) {
							widget.css('width', expandedWidth + 'px');
						} else {
							widget.css('width', defaultWidth + 'px');
						}
						// Remove inline height to let CSS handle it (CSS has !important)
						widget.css('height', '');
						// Scroll to bottom when widget opens
						setTimeout(function() {
							const messagesContainer = $('#knova-messages-container');
							if (messagesContainer.length) {
								scrollToBottom(messagesContainer.parent());
							}
						}, 100);
					}
				});
				isOpen = true;
				backBtn.show();
				input.focus();
			}
		}

		// Toggle button click - use event delegation to ensure it works
		$(document).off('click', '#knova-widget-toggle').on('click', '#knova-widget-toggle', function(e) {
			e.preventDefault();
			e.stopPropagation();
			toggleWidget();
			return false;
		});

		// Menu toggle functionality
		function toggleMenu() {
			if (isMenuOpen) {
				menuDropdown.stop(true, true).fadeOut(150);
				isMenuOpen = false;
			} else {
				// Close any other open menus first
				$('.knova-menu-dropdown').not(menuDropdown).fadeOut(150);
				menuDropdown.stop(true, true).fadeIn(150);
				isMenuOpen = true;
			}
		}

		function closeMenu() {
			if (isMenuOpen) {
				menuDropdown.fadeOut(150);
				isMenuOpen = false;
			}
		}

		// Expand/Collapse functionality (horizontal expansion)
		function expandWidget() {
			if (!isExpanded) {
				widget.addClass('knova-expanded');
				widget.animate({ 
					width: expandedWidth + 'px' 
				}, {
					duration: 300,
					easing: 'swing',
					queue: false
				});
				isExpanded = true;
				expandMenuItem.hide();
				collapseMenuItem.show();
				closeMenu();
			}
		}

		function collapseWidget() {
			if (isExpanded) {
				widget.removeClass('knova-expanded');
				widget.animate({ 
					width: defaultWidth + 'px' 
				}, {
					duration: 300,
					easing: 'swing',
					queue: false
				});
				isExpanded = false;
				expandMenuItem.show();
				collapseMenuItem.hide();
				closeMenu();
			}
		}

		// Menu button click - use event delegation to ensure it works even if element is dynamically added
		$(document).off('click', '#knova-menu-button').on('click', '#knova-menu-button', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			toggleMenu();
			return false;
		});

		// Expand menu item click - use event delegation
		$(document).off('click', '#knova-expand-menu-item').on('click', '#knova-expand-menu-item', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			expandWidget();
			return false;
		});

		// Collapse menu item click - use event delegation
		$(document).off('click', '#knova-collapse-menu-item').on('click', '#knova-collapse-menu-item', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			collapseWidget();
			return false;
		});

		// Clear chat functionality
		function clearChat() {
			// Check if SweetAlert2 is available
			if (typeof Swal === 'undefined') {
				// Fallback to native confirm if SweetAlert2 is not loaded
				if (!confirm('Are you sure you want to clear the chat history? This action cannot be undone.')) {
					return;
				}
				// Proceed with clearing if confirmed
				proceedWithClear();
				return;
			}

			Swal.fire({
				title: 'Clear Chat?',
				text: 'Are you sure you want to clear the chat history? This action cannot be undone.',
				icon: 'warning',
				showCancelButton: true,
				confirmButtonColor: '#dc3232',
				cancelButtonColor: '#6c757d',
				confirmButtonText: 'Yes, clear it!',
				cancelButtonText: 'Cancel',
				reverseButtons: true
			}).then((result) => {
				if (!result.isConfirmed) {
					return;
				}
				proceedWithClear();
			});
		}

		// Function to actually clear the chat
		function proceedWithClear() {

			// Clear displayed messages (keep only welcome message)
			messagesContainer.empty();
			
			// Add welcome message back
			const welcomeMessage = settings.welcomeMessage || 'Hello! How can I help you today?';
			const welcomeDiv = $('<div>').addClass('knova-message knova-message-assistant');
			const welcomeBubble = $('<div>').addClass('knova-message-bubble');
			welcomeBubble.html(welcomeMessage);
			welcomeDiv.append(welcomeBubble);
			messagesContainer.append(welcomeDiv);

			// Clear stored conversation
			if (settings.storageType === 'localStorage') {
				localStorage.removeItem('knova_conversation_' + sessionId);
			}

			// Clear database conversation if needed (via AJAX)
			$.ajax({
				url: settings.ajaxClearUrl || '/knova/ajax/clear-conversation',
				method: 'POST',
				data: JSON.stringify({
					nonce: settings.nonce,
					session_id: sessionId
				}),
				contentType: 'application/json',
				success: function(response) {
					// Scroll to bottom
					setTimeout(function() {
						scrollToBottom(messagesContainer.parent());
					}, 100);

					// Close menu
					closeMenu();

					// Show success message
					if (typeof Swal !== 'undefined') {
						Swal.fire({
							title: 'Chat Cleared!',
							text: 'Your chat history has been cleared.',
							icon: 'success',
							timer: 2000,
							showConfirmButton: false,
							toast: true,
							position: 'top-end'
						});
					}
				},
				error: function(xhr, status, error) {
					// Even if server call fails, local storage is already cleared
					// Scroll to bottom
					setTimeout(function() {
						scrollToBottom(messagesContainer.parent());
					}, 100);

					// Close menu
					closeMenu();

					// Show success message (local clear was successful)
					if (typeof Swal !== 'undefined') {
						Swal.fire({
							title: 'Chat Cleared!',
							text: 'Your chat history has been cleared.',
							icon: 'success',
							timer: 2000,
							showConfirmButton: false,
							toast: true,
							position: 'top-end'
						});
					}
				}
			});
		}

		// Clear chat menu item click - use event delegation to ensure it works
		$(document).on('click', '#knova-clear-chat-item', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			clearChat();
			return false;
		});

		// Close menu when clicking outside
		$(document).on('click.knova-menu', function(e) {
			if (isMenuOpen) {
				const target = $(e.target);
				// Don't close if clicking inside menu container, menu dropdown, or menu items (including clear chat)
				// Also don't close if clicking the menu button itself
				if (!target.closest('.knova-menu-container').length && 
				    !target.closest('.knova-menu-dropdown').length &&
				    !target.closest('.knova-menu-item').length &&
				    !target.closest('#knova-clear-chat-item').length &&
				    !target.closest('#knova-menu-button').length) {
					closeMenu();
				}
			}
		});

		// Back button click (minimize) - use event delegation
		$(document).off('click', '#knova-back-button').on('click', '#knova-back-button', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			toggleWidget();
			return false;
		});

		// Close button click - use event delegation to ensure it works even if element is dynamically added
		$(document).off('click', '#knova-close-button').on('click', '#knova-close-button', function(e) {
			e.preventDefault();
			e.stopPropagation();
			e.stopImmediatePropagation();
			toggleWidget();
			return false;
		});

		// Click outside to close - use namespace to avoid conflicts
		$(document).off('click.knova-outside').on('click.knova-outside', function(e) {
			if (isOpen) {
				const target = $(e.target);
				// Don't close if clicking inside the widget container
				if (!target.closest('#knova-widget-container').length) {
					toggleWidget();
				}
			}
		});

		// ESC key to close - use namespace to avoid conflicts
		$(document).off('keydown.knova-esc').on('keydown.knova-esc', function(e) {
			if (e.key === 'Escape' && isOpen) {
				toggleWidget();
			}
		});

		// Form submission - use event delegation
		$(document).off('submit', '#knova-input-form').on('submit', '#knova-input-form', function(e) {
			e.preventDefault();
			
			// Re-query elements in case they were recreated
			const formInput = $('#knova-input-field');
			const formSendBtn = $('#knova-send-button');
			const formMessagesContainer = $('#knova-messages-container');
			
			const message = formInput.val().trim();
			if (!message) return;

			// Disable input and button
			formInput.prop('disabled', true);
			formSendBtn.prop('disabled', true);

			// Add user message
			addMessage(formMessagesContainer, message, 'user');
			formInput.val('');

			// Show typing indicator
			toggleTypingIndicator(true);
			scrollToBottom(formMessagesContainer);

			// Get conversation from localStorage
			let conversation = [];
			if (settings.storageType === 'localStorage') {
				const stored = localStorage.getItem('knova_conversation_' + sessionId);
				if (stored) {
					try {
						conversation = JSON.parse(stored);
					} catch (e) {
						console.error('Error parsing conversation:', e);
						conversation = [];
					}
				}
			}

			// Build messages array - always start with system prompt and Q&A pairs
			let messages = [];
			
			// Add system prompt
			if (settings.systemPrompt) {
				messages.push({
					role: 'system',
					content: settings.systemPrompt
				});
			}

			// Add Q&A pairs
			if (settings.qaPairs && settings.qaPairs.length > 0) {
				settings.qaPairs.forEach(function(pair) {
					if (pair.question && pair.answer) {
						messages.push({
							role: 'user',
							content: pair.question
						});
						messages.push({
							role: 'assistant',
							content: pair.answer
						});
					}
				});
			}

			// Add existing conversation (excluding system messages)
			if (conversation.length > 0) {
				conversation.forEach(function(msg) {
					if (msg.role !== 'system') {
						messages.push(msg);
					}
				});
			}

			// Add current user message
			messages.push({
				role: 'user',
				content: message
			});

			// Send to server
			$.ajax({
				url: settings.ajaxUrl || '/knova/ajax/message',
				method: 'POST',
				data: JSON.stringify({
					nonce: settings.nonce,
					session_id: sessionId,
					messages: messages
				}),
				contentType: 'application/json',
				success: function(response) {
					// Hide typing indicator
					toggleTypingIndicator(false);

					// Check if response is valid
					if (!response || typeof response !== 'object') {
						console.error('Knova: Invalid response format', response);
						addMessage(formMessagesContainer, 'Invalid response from server. Please try again.', 'assistant');
						formInput.prop('disabled', false);
						formSendBtn.prop('disabled', false);
						return;
					}

					if (response.success === true || response.success === 'true') {
						const assistantMessage = response.data && response.data.message ? response.data.message : '';
						
						// Check if message is empty or blank
						if (!assistantMessage || assistantMessage.trim() === '') {
							const errorMsg = 'Sorry, I received an empty response. Please try again.';
							addMessage(formMessagesContainer, errorMsg, 'assistant');
							formInput.prop('disabled', false);
							formSendBtn.prop('disabled', false);
							return;
						}
						
						// Add assistant message with typewriter effect
						const messageDiv = $('<div>').addClass('knova-message knova-message-assistant');
						const bubble = $('<div>').addClass('knova-message-bubble');
						messageDiv.append(bubble);
						formMessagesContainer.append(messageDiv);

						typewriter(bubble, assistantMessage, function() {
							// Update conversation in localStorage (only save actual conversation, not Q&A context)
							if (settings.storageType === 'localStorage') {
								// Filter out Q&A pairs from saved conversation
								const conversationToSave = messages.filter(function(msg) {
									// Skip Q&A pairs (they're context, not conversation)
									if (msg.role === 'user') {
										const isQAPair = settings.qaPairs && settings.qaPairs.some(function(pair) {
											return pair.question === msg.content;
										});
										if (isQAPair) {
											return false;
										}
									}
									// Skip system messages
									if (msg.role === 'system') {
										return false;
									}
									return true;
								});
								
								// Add the new assistant message
								conversationToSave.push({
									role: 'assistant',
									content: assistantMessage
								});
								
								localStorage.setItem('knova_conversation_' + sessionId, JSON.stringify(conversationToSave));
							}
						});
					} else {
						// Error response - show the error message from server
						const errorMsg = response.data && response.data.message ? response.data.message : 
						                 (response.message || settings.errorMessage || 'Sorry, something went wrong. Please try again.');
						
						console.error('Knova API Error:', errorMsg, response);
						
						if (!errorMsg || errorMsg.trim() === '') {
							addMessage(formMessagesContainer, 'Sorry, something went wrong. Please try again.', 'assistant');
						} else {
							addMessage(formMessagesContainer, errorMsg, 'assistant');
						}
					}
				},
				error: function(xhr, statusText, error) {
					// Hide typing indicator
					toggleTypingIndicator(false);
					
					// Provide user-friendly messages for different error types
					let errorMsg = '';
					
					if (xhr.status === 0) {
						// Network error or connection refused
						errorMsg = 'Unable to connect to the server. Please check your internet connection and try again.';
					} else if (xhr.status === 404) {
						errorMsg = 'The request could not be found. Please refresh the page and try again.';
					} else if (xhr.status === 500) {
						errorMsg = 'Server error occurred. Please try again in a moment.';
					} else if (xhr.status === 503) {
						errorMsg = 'Service temporarily unavailable. Please try again in a few moments.';
					} else if (statusText === 'timeout') {
						errorMsg = 'The request timed out. Please check your connection and try again.';
					} else if (statusText === 'abort') {
						errorMsg = 'Request was cancelled. Please try again.';
					} else {
						// Use custom error message or default
						errorMsg = settings.errorMessage || 'Sorry, something went wrong. Please try again.';
					}
					
					addMessage(formMessagesContainer, errorMsg, 'assistant');
				},
				complete: function() {
					// Re-enable input and button
					formInput.prop('disabled', false);
					formSendBtn.prop('disabled', false);
					formInput.focus();
				}
			});
		});

		// Load conversation from localStorage (only actual conversation, not Q&A context)
		if (settings.storageType === 'localStorage') {
			const stored = localStorage.getItem('knova_conversation_' + sessionId);
			if (stored) {
				try {
					const conversation = JSON.parse(stored);
					// Only load messages that are actual conversation (not Q&A pairs used as context)
					// Q&A pairs are identified by being in the initial messages before any real user interaction
					let foundRealUserMessage = false;
					conversation.forEach(function(msg) {
						// Skip Q&A pairs (they're context, not displayed conversation)
						// Only show messages after the first real user message
						if (msg.role === 'user' && !foundRealUserMessage) {
							// Check if this is a Q&A pair question (it would match one of our Q&A questions)
							const isQAPair = settings.qaPairs && settings.qaPairs.some(function(pair) {
								return pair.question === msg.content;
							});
							if (!isQAPair) {
								foundRealUserMessage = true;
							}
						}
						
						// Only display messages after we've found a real user message
						if (foundRealUserMessage && (msg.role === 'user' || msg.role === 'assistant')) {
							addMessage(messagesContainer, msg.content, msg.role);
						}
					});
					// Scroll to bottom after loading conversation
					setTimeout(function() {
						scrollToBottom(messagesContainer.parent());
					}, 100);
				} catch (e) {
					console.error('Error loading conversation:', e);
					// Clear corrupted data
					localStorage.removeItem('knova_conversation_' + sessionId);
				}
			}
		}

		// Inquiry Form Handling
		const inquiryForm = $('#knova-inquiry-form');
		const inquiryFormElement = $('#knova-inquiry-form-element');
		const inquiryCancelBtn = $('#knova-inquiry-cancel');
		const showInquiryBtn = $('<button type="button" class="knova-show-inquiry-btn" style="display: none;">Get in Touch</button>');

		// Function to show inquiry form
		function showInquiryForm() {
			messagesContainer.hide();
			inquiryForm.slideDown(300);
			$('#knova-input-field').closest('.knova-widget-footer').hide();
			scrollToBottom(messagesContainer.parent());
		}

		// Function to hide inquiry form
		function hideInquiryForm() {
			inquiryForm.slideUp(300, function() {
				messagesContainer.show();
				$('#knova-input-field').closest('.knova-widget-footer').show();
			});
		}

		// Cancel inquiry form
		$(document).on('click', '#knova-inquiry-cancel', function(e) {
			e.preventDefault();
			hideInquiryForm();
		});

		// Submit inquiry form
		$(document).on('submit', '#knova-inquiry-form-element', function(e) {
			e.preventDefault();
			
			const name = $('#knova-inquiry-name').val().trim();
			const email = $('#knova-inquiry-email').val().trim();
			const phone = $('#knova-inquiry-phone').val().trim();
			const query = $('#knova-inquiry-query').val().trim();

			// Validate email or phone
			if (!email && !phone) {
				alert('Please provide either an email address or phone number.');
				return;
			}

			// Validate email format if provided
			if (email && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email)) {
				alert('Please provide a valid email address.');
				return;
			}

			// Get conversation summary from displayed messages
			let conversationSummary = '';
			messagesContainer.find('.knova-message').each(function() {
				const role = $(this).hasClass('knova-message-user') ? 'User' : 'Assistant';
				const content = $(this).find('.knova-message-bubble').text().trim();
				if (content) {
					conversationSummary += role + ': ' + content + '\n';
				}
			});

			// Disable submit button
			const submitBtn = $('#knova-inquiry-submit');
			submitBtn.prop('disabled', true).text('Submitting...');

			// Get CSRF token
			const nonce = settings.nonce || '';

			// Send inquiry
			$.ajax({
				url: settings.ajaxInquiryUrl || '/knova/ajax/save-inquiry',
				method: 'POST',
				contentType: 'application/json',
				data: JSON.stringify({
					nonce: nonce,
					session_id: sessionId,
					name: name,
					email: email,
					phone: phone,
					query: query || 'User inquiry from chatbot',
					conversation_summary: conversationSummary
				}),
				success: function(response) {
					if (response.success) {
						// Show success message
						addMessage(messagesContainer, response.data.message || 'Thank you! We\'ll get back to you soon.', 'assistant');
						hideInquiryForm();
						// Clear form
						inquiryFormElement[0].reset();
					} else {
						alert(response.data.message || 'An error occurred. Please try again.');
						submitBtn.prop('disabled', false).text('Submit');
					}
				},
				error: function() {
					alert('An error occurred. Please try again later.');
					submitBtn.prop('disabled', false).text('Submit');
				}
			});
		});


		// Also add a manual "Get in Touch" button that's always available
		// Add it to the menu or as a persistent button
		const getInTouchMenuItem = $('<button class="knova-menu-item" id="knova-get-in-touch-item"><svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M3 2H13C13.5523 2 14 2.44772 14 3V13C14 13.5523 13.5523 14 13 14H3C2.44772 14 2 13.5523 2 13V3C2 2.44772 2.44772 2 3 2Z" stroke="currentColor" stroke-width="1.5"/><path d="M5 5L8 8L11 5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg><span>Get in Touch</span></button>');
		getInTouchMenuItem.on('click', function() {
			showInquiryForm();
			closeMenu();
		});
		// Insert before the divider
		$('#knova-clear-chat-item').before(getInTouchMenuItem);
	}

	// Initialize when DOM is ready and also on page load
	// Use both ready and load to ensure widget HTML from hook_page_bottom is available
	$(document).ready(function() {
		// Try to initialize immediately
		initWidget();
		
		// Also try after a short delay in case widget HTML is added later
		setTimeout(function() {
			// Only initialize if not already initialized
			if ($('#knova-widget-container').length && !$('#knova-widget-container').data('knova-initialized')) {
				$('#knova-widget-container').data('knova-initialized', true);
				initWidget();
			}
		}, 100);
	});
	
	// Also initialize on window load as fallback
	$(window).on('load', function() {
		if ($('#knova-widget-container').length && !$('#knova-widget-container').data('knova-initialized')) {
			$('#knova-widget-container').data('knova-initialized', true);
			initWidget();
		}
	});

})(jQuery);

