<?php

namespace Drupal\knova\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\knova\Service\ApiHandler;
use Drupal\knova\Service\ConversationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * AJAX Controller for Knova.
 */
class AjaxController extends ControllerBase {

  /**
   * The API handler.
   *
   * @var \Drupal\knova\Service\ApiHandler
   */
  protected $apiHandler;

  /**
   * The conversation manager.
   *
   * @var \Drupal\knova\Service\ConversationManager
   */
  protected $conversationManager;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an AjaxController object.
   *
   * @param \Drupal\knova\Service\ApiHandler $api_handler
   *   The API handler.
   * @param \Drupal\knova\Service\ConversationManager $conversation_manager
   *   The conversation manager.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   */
  public function __construct(ApiHandler $api_handler, ConversationManager $conversation_manager, CsrfTokenGenerator $csrf_token) {
    $this->apiHandler = $api_handler;
    $this->conversationManager = $conversation_manager;
    $this->csrfToken = $csrf_token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static(
      $container->get('knova.api_handler'),
      $container->get('knova.conversation_manager'),
      $container->get('csrf_token')
    );
    $instance->logger = $container->get('logger.factory')->get('knova');
    return $instance;
  }

  /**
   * Send message handler.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function sendMessage(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error('Knova AJAX: Invalid JSON in request. Error: @error', [
        '@error' => json_last_error_msg(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // Verify CSRF token
    $token = $content['nonce'] ?? $request->request->get('nonce');
    if (!$this->csrfToken->validate($token, 'knova_ajax')) {
      $this->logger->warning('Knova AJAX: Invalid CSRF token');
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
      ], 403);
    }

    $messages = $content['messages'] ?? [];
    // Use Drupal session service
    $session = \Drupal::request()->getSession();
    $session_id = $content['session_id'] ?? ($session->has('knova_session_id') ? $session->get('knova_session_id') : $session->getId());

    if (empty($messages)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('No messages provided.')],
      ], 400);
    }

    // Sanitize messages
    $sanitized_messages = [];
    foreach ($messages as $message) {
      if (isset($message['role']) && isset($message['content'])) {
        $sanitized_messages[] = [
          'role' => htmlspecialchars($message['role'], ENT_QUOTES, 'UTF-8'),
          'content' => $message['content'], // Content is sent to API, no need to sanitize HTML
        ];
      }
    }

    // Send to API
    try {
      $response = $this->apiHandler->sendMessage($sanitized_messages);

      // Handle errors - API handler returns string for errors
      if (is_string($response)) {
        // It's an error message
        $this->logger->warning('Knova AJAX: API error - @message', [
          '@message' => $response,
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $response],
        ], 200); // Return 200 so it goes to success handler, not error handler
      }

      // Validate response is an array
      if (!is_array($response)) {
        $this->logger->error('Knova AJAX: Invalid API response type. Type: @type', [
          '@type' => gettype($response),
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $this->t('Invalid response from AI service. Please try again.')],
        ], 200); // Return 200 so it goes to success handler
      }

      // Extract assistant message
      $assistant_message = '';
      if (isset($response['choices'][0]['message']['content'])) {
        $assistant_message = $response['choices'][0]['message']['content'];
      }
      
      if (empty($assistant_message)) {
        $this->logger->warning('Knova AJAX: Empty assistant message in response', [
          'response' => $response,
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $this->t('The AI service returned an empty response. Please try again.')],
        ], 200); // Return 200 so it goes to success handler
      }

      // Add intelligent links if Q&A pairs have URLs
      $assistant_message = $this->addIntelligentLinks($assistant_message, $sanitized_messages);

      return new JsonResponse([
        'success' => TRUE,
        'data' => ['message' => $assistant_message],
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Knova AJAX: Exception in sendMessage - @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Server error occurred. Please try again in a moment.')],
      ], 200); // Return 200 so it goes to success handler with error message
    }
  }

  /**
   * Clear conversation handler.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function clearConversation(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // Verify CSRF token
    $token = $content['nonce'] ?? $request->request->get('nonce');
    if (!$this->csrfToken->validate($token, 'knova_ajax')) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
      ], 403);
    }

    // Use Drupal session service
    $session = \Drupal::request()->getSession();
    $session_id = $content['session_id'] ?? ($session->has('knova_session_id') ? $session->get('knova_session_id') : $session->getId());

    $this->conversationManager->clearConversation($session_id);

    return new JsonResponse([
      'success' => TRUE,
      'data' => ['message' => $this->t('Conversation cleared.')],
    ]);
  }

  /**
   * Add intelligent links to response.
   *
   * @param string $message
   *   The message.
   * @param array $context_messages
   *   Context messages.
   *
   * @return string
   *   Message with links.
   */
  private function addIntelligentLinks($message, array $context_messages) {
    $settings_manager = \Drupal::service('knova.settings_manager');
    $settings = $settings_manager->getSettings();
    $qa_pairs = $settings['qa_pairs'] ?? [];

    // Check if message references any Q&A content
    foreach ($qa_pairs as $pair) {
      if (!empty($pair['url']) && !empty($pair['question'])) {
        // Simple keyword matching - if message contains words from question
        $question_words = explode(' ', strtolower($pair['question']));
        $message_lower = strtolower($message);
        $match_count = 0;

        foreach ($question_words as $word) {
          if (strlen($word) > 3 && strpos($message_lower, $word) !== FALSE) {
            $match_count++;
          }
        }

        // If significant match and URL not already in message
        if ($match_count >= 2 && strpos($message, $pair['url']) === FALSE) {
          $message .= "\n\nFor more information, visit: " . $pair['url'];
          break;
        }
      }
    }

    return $message;
  }

  /**
   * Save user inquiry/lead.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function saveInquiry(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error('Knova AJAX: Invalid JSON in inquiry request. Error: @error', [
        '@error' => json_last_error_msg(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // Verify CSRF token
    $token = $content['nonce'] ?? $request->request->get('nonce');
    if (!$this->csrfToken->validate($token, 'knova_ajax')) {
      $this->logger->warning('Knova AJAX: Invalid CSRF token for inquiry');
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
      ], 403);
    }

    // Get inquiry data
    $session = \Drupal::request()->getSession();
    $session_id = $content['session_id'] ?? ($session->has('knova_session_id') ? $session->get('knova_session_id') : $session->getId());
    
    $name = isset($content['name']) ? trim($content['name']) : '';
    $email = isset($content['email']) ? trim($content['email']) : '';
    $phone = isset($content['phone']) ? trim($content['phone']) : '';
    $query = isset($content['query']) ? trim($content['query']) : '';
    $conversation_summary = isset($content['conversation_summary']) ? trim($content['conversation_summary']) : '';

    // Validate required fields (at least email or phone)
    if (empty($email) && empty($phone)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Please provide either an email address or phone number.')],
      ], 200);
    }

    // Validate email format if provided
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Please provide a valid email address.')],
      ], 200);
    }

    try {
      // Save to database
      $database = \Drupal::database();
      $now = date('Y-m-d H:i:s', \Drupal::time()->getRequestTime());

      $fields = [
        'session_id' => $session_id,
        'name' => $name,
        'email' => $email,
        'phone' => $phone,
        'query' => $query,
        'conversation_summary' => $conversation_summary,
        'status' => 'new',
        'created_at' => $now,
        'updated_at' => $now,
      ];

      $database->insert('knova_inquiries')
        ->fields($fields)
        ->execute();

      $this->logger->info('Knova: New inquiry saved. Email: @email, Phone: @phone', [
        '@email' => $email ?: 'N/A',
        '@phone' => $phone ?: 'N/A',
      ]);

      // TODO: Send email notification if configured
      // You can add email functionality here

      return new JsonResponse([
        'success' => TRUE,
        'data' => ['message' => $this->t('Thank you! We\'ll get back to you soon.')],
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Knova AJAX: Exception saving inquiry - @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('An error occurred. Please try again later.')],
      ], 200);
    }
  }

}

