<?php

namespace Drupal\knova\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\knova\Service\SettingsManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Knova.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The settings manager.
   *
   * @var \Drupal\knova\Service\SettingsManager
   */
  protected $settingsManager;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\knova\Service\SettingsManager $settings_manager
   *   The settings manager.
   */
  public function __construct(SettingsManager $settings_manager) {
    $this->settingsManager = $settings_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('knova.settings_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['knova.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'knova_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('knova.settings');
    // Force refresh to get latest settings (bypass cache)
    $settings = $this->settingsManager->getSettings(TRUE);

    $form['#attached']['library'][] = 'knova/admin';
    $form['#attached']['library'][] = 'core/drupal.ajax';

    // Vertical tabs
    $form['tabs'] = [
      '#type' => 'vertical_tabs',
      '#default_tab' => 'edit-general',
    ];

    // General tab
    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General'),
      '#group' => 'tabs',
    ];

    $form['general']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Widget'),
      '#default_value' => $settings['enabled'] ?? TRUE,
      '#description' => $this->t('Enable or disable the chatbot widget on your site.'),
    ];

    $form['general']['position'] = [
      '#type' => 'select',
      '#title' => $this->t('Widget Position'),
      '#options' => [
        'right' => $this->t('Right'),
        'left' => $this->t('Left'),
      ],
      '#default_value' => $settings['position'] ?? 'right',
      '#description' => $this->t('Choose which corner the widget appears in.'),
    ];

    $form['general']['distance_bottom'] = [
      '#type' => 'number',
      '#title' => $this->t('Distance from Bottom (px)'),
      '#default_value' => $settings['distance_bottom'] ?? 20,
      '#min' => 0,
      '#description' => $this->t('Distance of the widget from the bottom of the screen in pixels.'),
    ];

    $form['general']['distance_side'] = [
      '#type' => 'number',
      '#title' => $this->t('Distance from Side (px)'),
      '#default_value' => $settings['distance_side'] ?? 20,
      '#min' => 0,
      '#description' => $this->t('Distance of the widget from the side of the screen in pixels.'),
    ];

    $form['general']['mobile_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable on Mobile'),
      '#default_value' => $settings['mobile_enabled'] ?? TRUE,
      '#description' => $this->t('Show the widget on mobile devices.'),
    ];

    // Appearance tab
    $form['appearance'] = [
      '#type' => 'details',
      '#title' => $this->t('Appearance'),
      '#group' => 'tabs',
    ];

    $form['appearance']['primary_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Header & Toggle Button Color'),
      '#default_value' => $settings['primary_color'] ?? '#0073aa',
      '#description' => $this->t('The main color used for the widget header, toggle button, and user message bubbles. This is the primary brand color of your chatbot.'),
    ];

    $form['appearance']['secondary_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Widget Background Color'),
      '#default_value' => $settings['secondary_color'] ?? '#005177',
      '#description' => $this->t('The background color for the main widget body area where conversations are displayed.'),
    ];

    $form['appearance']['text_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Message Text Color'),
      '#default_value' => $settings['text_color'] ?? '#333333',
      '#description' => $this->t('The color used for all message text content in the conversation area.'),
    ];

    $form['appearance']['accent_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Button & Action Color'),
      '#default_value' => $settings['accent_color'] ?? '#0073aa',
      '#description' => $this->t('The color used for interactive elements like the send button, submit buttons, and focus highlights.'),
    ];

    $form['appearance']['widget_background_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Widget Container Background'),
      '#default_value' => $settings['widget_background_color'] ?? '#ffffff',
      '#description' => $this->t('The background color for the main widget container (the outer box that contains the entire chat widget).'),
    ];

    $form['appearance']['user_message_bg_color'] = [
      '#type' => 'color',
      '#title' => $this->t('User Message Bubble Background'),
      '#default_value' => $settings['user_message_bg_color'] ?? '#0073aa',
      '#description' => $this->t('The background color for user message bubbles (messages sent by the visitor).'),
    ];

    $form['appearance']['assistant_message_bg_color'] = [
      '#type' => 'color',
      '#title' => $this->t('Assistant Message Bubble Background'),
      '#default_value' => $settings['assistant_message_bg_color'] ?? '#ffffff',
      '#description' => $this->t('The background color for assistant message bubbles (messages from the chatbot).'),
    ];

    $form['appearance']['border_radius'] = [
      '#type' => 'number',
      '#title' => $this->t('Border Radius (px)'),
      '#default_value' => $settings['border_radius'] ?? 12,
      '#min' => 0,
      '#description' => $this->t('Border radius for rounded corners.'),
    ];

    $form['appearance']['widget_width_desktop'] = [
      '#type' => 'number',
      '#title' => $this->t('Widget Width - Desktop (px)'),
      '#default_value' => $settings['widget_width_desktop'] ?? 380,
      '#min' => 300,
      '#max' => 600,
      '#description' => $this->t('Maximum width of the widget on desktop devices.'),
    ];

    $form['appearance']['widget_height_max'] = [
      '#type' => 'number',
      '#title' => $this->t('Widget Height - Max (px)'),
      '#default_value' => $settings['widget_height_max'] ?? 400,
      '#min' => 400,
      '#max' => 800,
      '#description' => $this->t('Maximum height of the widget.'),
    ];

    $form['appearance']['logo_url'] = [
      '#type' => 'url',
      '#title' => $this->t('Logo URL'),
      '#default_value' => $settings['logo_url'] ?? '',
      '#description' => $this->t('URL of the logo image to display in the widget header.'),
    ];

    $form['appearance']['title_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title Text'),
      '#default_value' => $settings['title_text'] ?? 'Knova - Intelligence',
      '#description' => $this->t('Title displayed in the widget header.'),
    ];

    $form['appearance']['tagline_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Tagline Text'),
      '#default_value' => $settings['tagline_text'] ?? 'AI-Powered Assistant',
      '#description' => $this->t('Tagline displayed below the title.'),
    ];

    $form['appearance']['welcome_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Welcome Message'),
      '#default_value' => $settings['welcome_message'] ?? 'Hello! I\'m Knova, your intelligent AI assistant. How can I help you today?',
      '#description' => $this->t('Initial message shown when the widget opens.'),
    ];

    $form['appearance']['input_placeholder'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Input Placeholder'),
      '#default_value' => $settings['input_placeholder'] ?? 'Type your message...',
      '#description' => $this->t('Placeholder text for the input field.'),
    ];

    // AI Settings tab
    $form['ai'] = [
      '#type' => 'details',
      '#title' => $this->t('AI Settings'),
      '#group' => 'tabs',
      '#tree' => TRUE,
    ];

    $form['ai']['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#options' => [
        'gpt-4' => 'GPT-4',
        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
      ],
      '#default_value' => $settings['model'] ?? 'gpt-3.5-turbo',
      '#description' => $this->t('Select the AI model to use.'),
    ];

    $form['ai']['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#default_value' => $settings['temperature'] ?? 0.7,
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
      '#description' => $this->t('Controls randomness. Lower values make responses more focused and deterministic. (0-2)'),
    ];

    $form['ai']['max_tokens'] = [
      '#type' => 'number',
      '#title' => $this->t('Max Tokens'),
      '#default_value' => $settings['max_tokens'] ?? 256,
      '#min' => 1,
      '#max' => 4096,
      '#description' => $this->t('Maximum number of tokens in the response. (1-4096)'),
    ];

    $form['ai']['top_p'] = [
      '#type' => 'number',
      '#title' => $this->t('Top P'),
      '#default_value' => $settings['top_p'] ?? 1,
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.1,
      '#description' => $this->t('Controls diversity via nucleus sampling. (0-1)'),
    ];

    $form['ai']['frequency_penalty'] = [
      '#type' => 'number',
      '#title' => $this->t('Frequency Penalty'),
      '#default_value' => $settings['frequency_penalty'] ?? 0,
      '#min' => -2,
      '#max' => 2,
      '#step' => 0.1,
      '#description' => $this->t('Reduces likelihood of repeating tokens. (-2 to 2)'),
    ];

    $form['ai']['presence_penalty'] = [
      '#type' => 'number',
      '#title' => $this->t('Presence Penalty'),
      '#default_value' => $settings['presence_penalty'] ?? 0,
      '#min' => -2,
      '#max' => 2,
      '#step' => 0.1,
      '#description' => $this->t('Increases likelihood of talking about new topics. (-2 to 2)'),
    ];

    $form['ai']['endpoint'] = [
      '#type' => 'url',
      '#title' => $this->t('API Endpoint'),
      '#default_value' => $settings['endpoint'] ?? 'https://api.openai.com/v1/chat/completions',
      '#description' => $this->t('API endpoint URL for chat completions.'),
    ];

    // Create masked display of API key if one exists (like WordPress)
    $existing_api_key = $settings['api_key'] ?? '';
    $api_key_display = '';
    if (!empty($existing_api_key)) {
      $key_length = strlen($existing_api_key);
      // Show first 7 chars and last 4 chars, mask the middle (like WordPress)
      if ($key_length > 11) {
        $api_key_display = substr($existing_api_key, 0, 7) . str_repeat('•', max(8, $key_length - 11)) . substr($existing_api_key, -4);
      } else {
        // For very short keys, just mask everything except first 3 and last 2
        $api_key_display = substr($existing_api_key, 0, 3) . str_repeat('•', max(4, $key_length - 5)) . substr($existing_api_key, -2);
      }
    }
    
    $form['ai']['api_key'] = [
      '#type' => 'textarea',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('API key for authentication. Enter a new value to update, or leave as is to keep the current key.'),
      '#default_value' => $api_key_display,
      '#rows' => 2,
      '#attributes' => [
        'autocomplete' => 'new-password',
        'class' => ['api-key-field'],
        'data-has-existing-key' => !empty($existing_api_key) ? '1' : '0',
        'style' => 'font-family: monospace;',
      ],
    ];

    $form['ai']['system_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt'),
      '#default_value' => $settings['system_prompt'] ?? '',
      '#rows' => 10,
      '#description' => $this->t('System prompt that defines the AI assistant\'s behavior and personality.'),
    ];

    // Q&A Pairs
    $form['ai']['qa_pairs'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Q&A Pairs'),
      '#description' => $this->t('Add question and answer pairs to train the chatbot. These will be used as context for the AI.'),
      '#tree' => TRUE,
    ];

    $qa_pairs = $settings['qa_pairs'] ?? [];
    
    // Ensure qa_pairs is an array
    if (!is_array($qa_pairs)) {
      $qa_pairs = [];
    }
    
    // Log for debugging
    \Drupal::logger('knova')->debug('Loading @count Q&A pairs for form display', [
      '@count' => count($qa_pairs),
    ]);
    
    // Store initial count for JavaScript
    $form['ai']['qa_pairs']['#attached']['drupalSettings']['knova_qa_pairs_count'] = count($qa_pairs);
    
    $form['ai']['qa_pairs']['pairs'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'knova-qa-pairs-container'],
      '#tree' => TRUE,
    ];

    // Build existing pairs - use sequential numeric indices starting from 0
    // This ensures proper form handling and matches what JavaScript expects
    $pair_index = 0;
    foreach ($qa_pairs as $pair) {
      // Only process valid array pairs
      if (is_array($pair)) {
        // Display all pairs that have at least a question or answer
        // This allows users to see and edit all saved pairs
        if (!empty($pair['question']) || !empty($pair['answer'])) {
          $form['ai']['qa_pairs']['pairs'][$pair_index] = $this->buildQaPairForm($pair_index, $pair);
          $pair_index++;
        }
      }
    }

    $form['ai']['qa_pairs']['add_pair'] = [
      '#type' => 'button',
      '#value' => $this->t('Add Q&A Pair'),
      '#attributes' => [
        'class' => ['button', 'knova-add-qa-pair'],
        'type' => 'button',
      ],
    ];

    // Advanced tab
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
      '#group' => 'tabs',
    ];

    $form['advanced']['conversation_storage'] = [
      '#type' => 'select',
      '#title' => $this->t('Conversation Storage'),
      '#options' => [
        'localStorage' => $this->t('LocalStorage (Browser)'),
        'database' => $this->t('Database'),
        'session' => $this->t('Session'),
      ],
      '#default_value' => $settings['conversation_storage'] ?? 'localStorage',
      '#description' => $this->t('Where to store conversation history.'),
    ];

    $form['advanced']['rate_limit_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Rate Limiting'),
      '#default_value' => $settings['rate_limit_enabled'] ?? TRUE,
      '#description' => $this->t('Limit the number of requests per user.'),
    ];

    $form['advanced']['rate_limit_requests'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate Limit - Requests'),
      '#default_value' => $settings['rate_limit_requests'] ?? 10,
      '#min' => 1,
      '#states' => [
        'visible' => [
          ':input[name="rate_limit_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['advanced']['rate_limit_period'] = [
      '#type' => 'number',
      '#title' => $this->t('Rate Limit - Period (seconds)'),
      '#default_value' => $settings['rate_limit_period'] ?? 60,
      '#min' => 1,
      '#states' => [
        'visible' => [
          ':input[name="rate_limit_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['advanced']['error_message'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Error Message'),
      '#default_value' => $settings['error_message'] ?? 'Sorry, something went wrong. Please try again.',
      '#description' => $this->t('Default error message shown to users.'),
    ];

    $form['advanced']['custom_css'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom CSS'),
      '#default_value' => $settings['custom_css'] ?? '',
      '#rows' => 10,
      '#description' => $this->t('Add custom CSS to style the widget.'),
    ];

    // Internal Analytics tab
    $form['analytics'] = [
      '#type' => 'details',
      '#title' => $this->t('Internal Analytics'),
      '#group' => 'tabs',
    ];

    $form['analytics']['coming_soon'] = [
      '#type' => 'markup',
      '#markup' => '<div class="knova-coming-soon">
        <h3>' . $this->t('Coming Soon') . '</h3>
        <p>' . $this->t('Internal analytics features are currently under development. This section will provide insights into chatbot usage, conversation statistics, user engagement metrics, and performance analytics.') . '</p>
        <p><strong>' . $this->t('Planned features:') . '</strong></p>
        <ul>
          <li>' . $this->t('Conversation statistics and trends') . '</li>
          <li>' . $this->t('User engagement metrics') . '</li>
          <li>' . $this->t('Most asked questions') . '</li>
          <li>' . $this->t('Response time analytics') . '</li>
          <li>' . $this->t('Usage reports and exports') . '</li>
        </ul>
      </div>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Build Q&A pair form element.
   *
   * @param int $index
   *   Index of the pair.
   * @param array $pair
   *   Existing pair data.
   *
   * @return array
   *   Form element.
   */
  private function buildQaPairForm($index, array $pair = []) {
    return [
      '#type' => 'container',
      '#attributes' => ['class' => ['knova-qa-pair', 'form-wrapper']],
      '#tree' => TRUE,
      'question' => [
        '#type' => 'textfield',
        '#title' => $this->t('Question'),
        '#default_value' => $pair['question'] ?? '',
        '#required' => FALSE, // Don't require, we'll validate in submit
      ],
      'answer' => [
        '#type' => 'textarea',
        '#title' => $this->t('Answer'),
        '#default_value' => $pair['answer'] ?? '',
        '#required' => FALSE, // Don't require, we'll validate in submit
        '#rows' => 3,
      ],
      'url' => [
        '#type' => 'url',
        '#title' => $this->t('URL (optional)'),
        '#default_value' => $pair['url'] ?? '',
        '#description' => $this->t('Optional URL to include in the answer.'),
        '#required' => FALSE,
      ],
      'remove' => [
        '#type' => 'button',
        '#value' => $this->t('Remove'),
        '#attributes' => [
          'class' => ['button', 'knova-remove-qa-pair'],
          'type' => 'button',
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();
    $config = $this->configFactory()->getEditable('knova.settings');

    // Get existing API key directly from config (not cached)
    $existing_api_key = $config->get('api_key') ?? '';

    // Handle API key - check if user entered a new value
    $api_key_value = $form_state->getValue(['ai', 'api_key']);
    
    // If empty from form values, check user input (raw POST data)
    if (empty($api_key_value)) {
      $user_input = $form_state->getUserInput();
      $api_key_value = $user_input['ai']['api_key'] ?? NULL;
    }
    
    // Check if the submitted value is the masked display (user didn't change it)
    $masked_pattern = '';
    if (!empty($existing_api_key)) {
      $key_length = strlen($existing_api_key);
      if ($key_length > 11) {
        $masked_pattern = substr($existing_api_key, 0, 7) . str_repeat('•', max(8, $key_length - 11)) . substr($existing_api_key, -4);
      } else {
        $masked_pattern = substr($existing_api_key, 0, 3) . str_repeat('•', max(4, $key_length - 5)) . substr($existing_api_key, -2);
      }
    }
    
    // Save API key: if new value provided and different from masked, use it; otherwise preserve existing
    if (!empty($api_key_value) && is_string($api_key_value)) {
      $trimmed_key = trim($api_key_value);
      
      // If the value matches the masked pattern, user didn't change it - preserve existing
      if (!empty($masked_pattern) && $trimmed_key === $masked_pattern) {
        // User left the masked value unchanged, preserve existing key
        $config->set('api_key', $existing_api_key);
        \Drupal::logger('knova')->info('API key preserved (masked value unchanged)');
      } elseif (!empty($trimmed_key)) {
        // New value provided, save it
        $config->set('api_key', $trimmed_key);
        \Drupal::logger('knova')->info('API key updated (length: @length)', [
          '@length' => strlen($trimmed_key),
        ]);
      } else {
        // Empty string provided, clear the key
        $config->set('api_key', '');
        \Drupal::logger('knova')->info('API key cleared');
      }
    } else {
      // No value provided, preserve existing API key
      $config->set('api_key', $existing_api_key);
      if (!empty($existing_api_key)) {
        \Drupal::logger('knova')->info('API key preserved (no new value provided)');
      }
    }

    // Handle Q&A pairs from form values
    // Try multiple paths to get the values due to nested tree structure
    $qa_pairs_value = $form_state->getValue(['ai', 'qa_pairs', 'pairs'], []);
    
    // Also try getting from the full values array
    $all_values = $form_state->getValues();
    if (empty($qa_pairs_value) && isset($all_values['ai']['qa_pairs']['pairs'])) {
      $qa_pairs_value = $all_values['ai']['qa_pairs']['pairs'];
    }
    
    // Log what we got from form values
    \Drupal::logger('knova')->debug('Q&A pairs from form values: @count items, type: @type', [
      '@count' => is_array($qa_pairs_value) ? count($qa_pairs_value) : 0,
      '@type' => gettype($qa_pairs_value),
    ]);
    
    // Fallback to user input if form values are empty
    if (empty($qa_pairs_value) || !is_array($qa_pairs_value)) {
      $user_input = $form_state->getUserInput();
      $qa_pairs_value = $user_input['ai']['qa_pairs']['pairs'] ?? [];
      \Drupal::logger('knova')->debug('Q&A pairs from user input: @count items', [
        '@count' => is_array($qa_pairs_value) ? count($qa_pairs_value) : 0,
      ]);
      
      // Log the structure of user input for debugging
      if (!empty($user_input['ai']['qa_pairs']['pairs'])) {
        \Drupal::logger('knova')->debug('User input structure: @keys', [
          '@keys' => implode(', ', array_keys($user_input['ai']['qa_pairs']['pairs'])),
        ]);
      }
    }
    
    // Ensure we have an array
    if (!is_array($qa_pairs_value)) {
      $qa_pairs_value = [];
    }
    
    $pairs = [];
    if (!empty($qa_pairs_value) && is_array($qa_pairs_value)) {
      foreach ($qa_pairs_value as $index => $pair) {
        // Skip non-array elements (buttons, form elements, etc.)
        if (!is_array($pair)) {
          \Drupal::logger('knova')->debug('Skipping non-array Q&A pair at index @index', ['@index' => $index]);
          continue;
        }
        
        // Skip if this is a form element (has #type) or remove button
        if (isset($pair['#type']) || isset($pair['remove'])) {
          \Drupal::logger('knova')->debug('Skipping form element Q&A pair at index @index', ['@index' => $index]);
          continue;
        }
        
        // Extract and trim values
        $question = isset($pair['question']) ? trim((string) $pair['question']) : '';
        $answer = isset($pair['answer']) ? trim((string) $pair['answer']) : '';
        $url = isset($pair['url']) ? trim((string) $pair['url']) : '';
        
        \Drupal::logger('knova')->debug('Processing Q&A pair @index: question length=@qlen, answer length=@alen', [
          '@index' => $index,
          '@qlen' => strlen($question),
          '@alen' => strlen($answer),
        ]);
        
        // Only save pairs with both question and answer (required fields)
        if (!empty($question) && !empty($answer)) {
          $pairs[] = [
            'question' => $question,
            'answer' => $answer,
            'url' => $url,
          ];
          \Drupal::logger('knova')->debug('Added Q&A pair @index to save list', ['@index' => $index]);
        } else {
          \Drupal::logger('knova')->debug('Skipping Q&A pair @index: missing question or answer (q: "@q", a: "@a")', [
            '@index' => $index,
            '@q' => substr($question, 0, 20),
            '@a' => substr($answer, 0, 20),
          ]);
        }
      }
    }
    
    // Save Q&A pairs - Drupal config will handle the sequence properly
    $config->set('qa_pairs', $pairs);
    
    // Log for debugging
    \Drupal::logger('knova')->info('Saved @count Q&A pairs to configuration', [
      '@count' => count($pairs),
    ]);
    
    // Verify what was actually saved
    $saved_pairs = $config->get('qa_pairs');
    \Drupal::logger('knova')->debug('Verified saved Q&A pairs: @count items', [
      '@count' => is_array($saved_pairs) ? count($saved_pairs) : 0,
    ]);

    // Handle AI settings from nested structure
    if (isset($values['ai']) && is_array($values['ai'])) {
      $ai_settings = [
        'system_prompt',
        'model',
        'temperature',
        'max_tokens',
        'top_p',
        'frequency_penalty',
        'presence_penalty',
        'endpoint',
      ];
      
      foreach ($ai_settings as $setting) {
        if (isset($values['ai'][$setting])) {
          $config->set($setting, $values['ai'][$setting]);
        }
      }
    }

    // Handle appearance settings from nested structure
    $appearance_settings = [
      'primary_color',
      'secondary_color',
      'text_color',
      'accent_color',
      'widget_background_color',
      'user_message_bg_color',
      'assistant_message_bg_color',
      'border_radius',
      'widget_width_desktop',
      'widget_width_mobile',
      'widget_height_max',
      'logo_url',
      'title_text',
      'tagline_text',
      'welcome_message',
      'input_placeholder',
      'font_family',
    ];
    
    foreach ($appearance_settings as $setting) {
      // Check nested structure first (if #tree is used)
      if (isset($values['appearance'][$setting])) {
        $config->set($setting, $values['appearance'][$setting]);
      }
      // Fallback to top-level (if not using #tree)
      elseif (isset($values[$setting])) {
        $config->set($setting, $values[$setting]);
      }
    }

    // Handle general settings from nested structure
    $general_settings = [
      'enabled',
      'position',
      'distance_bottom',
      'distance_side',
      'mobile_enabled',
    ];
    
    foreach ($general_settings as $setting) {
      // Check nested structure first (if #tree is used)
      if (isset($values['general'][$setting])) {
        $config->set($setting, $values['general'][$setting]);
      }
      // Fallback to top-level (if not using #tree)
      elseif (isset($values[$setting])) {
        $config->set($setting, $values[$setting]);
      }
    }

    // Handle advanced settings from nested structure
    $advanced_settings = [
      'conversation_storage',
      'rate_limit_enabled',
      'rate_limit_requests',
      'rate_limit_period',
      'error_message',
      'debug_mode',
      'custom_css',
    ];
    
    foreach ($advanced_settings as $setting) {
      // Check nested structure first (if #tree is used)
      if (isset($values['advanced'][$setting])) {
        $config->set($setting, $values['advanced'][$setting]);
      }
      // Fallback to top-level (if not using #tree)
      elseif (isset($values[$setting])) {
        $config->set($setting, $values[$setting]);
      }
    }

    // Save all other top-level settings (fallback for any settings not in nested structures)
    $skip_keys = [
      'submit',
      'form_build_id',
      'form_token',
      'form_id',
      'op',
      'api_key',
      'ai',
      'appearance',
      'general',
      'advanced',
      'analytics',
      'tabs',
    ];
    
    foreach ($values as $key => $value) {
      if (!in_array($key, $skip_keys) && !is_array($value)) {
        $config->set($key, $value);
      }
    }

    // Save configuration
    $config->save();

    // Clear cache efficiently - must clear both cache backend and tags
    // This ensures the API handler gets fresh settings immediately
    $this->settingsManager->clearCache();
    \Drupal::service('cache.default')->delete('knova:settings');
    \Drupal::service('cache_tags.invalidator')->invalidateTags(['config:knova.settings']);
    
    // Verify API key was saved correctly (for debugging)
    $saved_key = $config->get('api_key');
    if (!empty($saved_key)) {
      \Drupal::logger('knova')->debug('API key verification after save: Key length is @length, starts with @prefix', [
        '@length' => strlen($saved_key),
        '@prefix' => substr($saved_key, 0, 7),
      ]);
    } else {
      \Drupal::logger('knova')->warning('API key verification after save: Key is empty');
    }
    
    // Verify Q&A pairs were saved correctly
    $saved_qa_pairs = $config->get('qa_pairs');
    \Drupal::logger('knova')->info('Q&A pairs verification after save: @count pairs saved', [
      '@count' => is_array($saved_qa_pairs) ? count($saved_qa_pairs) : 0,
    ]);
    
    // Show success message
    $message_parts = [];
    if (!empty($pairs)) {
      $message_parts[] = $this->formatPlural(count($pairs), '1 Q&A pair saved.', '@count Q&A pairs saved.');
    }
    if (!empty($api_key_value) && is_string($api_key_value) && strlen(trim($api_key_value)) > 0) {
      $message_parts[] = $this->t('API key updated.');
    }
    if (empty($message_parts)) {
      $message_parts[] = $this->t('Settings saved.');
    }
    $this->messenger()->addStatus(implode(' ', $message_parts));

    parent::submitForm($form, $form_state);
  }

}

