<?php

namespace Drupal\knova\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Conversation Manager service for Knova.
 */
class ConversationManager {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The settings manager.
   *
   * @var \Drupal\knova\Service\SettingsManager
   */
  protected $settingsManager;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a ConversationManager object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\knova\Service\SettingsManager $settings_manager
   *   The settings manager.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   */
  public function __construct(Connection $database, SettingsManager $settings_manager, AccountProxyInterface $current_user) {
    $this->database = $database;
    $this->settingsManager = $settings_manager;
    $this->currentUser = $current_user;
  }

  /**
   * Get conversation messages.
   *
   * @param string $session_id
   *   Session ID.
   *
   * @return array
   *   Messages array.
   */
  public function getMessages($session_id) {
    $settings = $this->settingsManager->getSettings();

    if ($settings['conversation_storage'] === 'database') {
      return $this->getMessagesFromDatabase($session_id);
    }

    // For localStorage and session, return empty (handled by JS)
    return [];
  }

  /**
   * Save conversation messages.
   *
   * @param string $session_id
   *   Session ID.
   * @param array $messages
   *   Messages array.
   *
   * @return bool
   *   TRUE on success, FALSE otherwise.
   */
  public function saveMessages($session_id, array $messages) {
    $settings = $this->settingsManager->getSettings();

    if ($settings['conversation_storage'] === 'database') {
      return $this->saveMessagesToDatabase($session_id, $messages);
    }

    // For localStorage and session, handled by JS
    return TRUE;
  }

  /**
   * Get messages from database.
   *
   * @param string $session_id
   *   Session ID.
   *
   * @return array
   *   Messages array.
   */
  private function getMessagesFromDatabase($session_id) {
    $result = $this->database->select('knova_conversations', 'kc')
      ->fields('kc', ['messages'])
      ->condition('session_id', $session_id)
      ->orderBy('updated_at', 'DESC')
      ->range(0, 1)
      ->execute()
      ->fetchField();

    if ($result) {
      $messages = json_decode($result, TRUE);
      // Validate JSON decode
      if (json_last_error() !== JSON_ERROR_NONE || !is_array($messages)) {
        return [];
      }
      return $messages;
    }

    return [];
  }

  /**
   * Save messages to database.
   *
   * @param string $session_id
   *   Session ID.
   * @param array $messages
   *   Messages array.
   *
   * @return bool
   *   TRUE on success, FALSE otherwise.
   */
  private function saveMessagesToDatabase($session_id, array $messages) {
    $user_id = $this->currentUser->id();
    $messages_json = json_encode($messages);
    // Use Drupal's time service for proper timezone handling
    $now = date('Y-m-d H:i:s', \Drupal::time()->getRequestTime());

    // Check if session exists
    $existing = $this->database->select('knova_conversations', 'kc')
      ->fields('kc', ['id'])
      ->condition('session_id', $session_id)
      ->execute()
      ->fetchField();

    if ($existing) {
      // Update existing
      return $this->database->update('knova_conversations')
        ->fields([
          'user_id' => $user_id,
          'messages' => $messages_json,
          'updated_at' => $now,
        ])
        ->condition('session_id', $session_id)
        ->execute() !== NULL;
    }
    else {
      // Insert new
      return $this->database->insert('knova_conversations')
        ->fields([
          'session_id' => $session_id,
          'user_id' => $user_id,
          'messages' => $messages_json,
          'created_at' => $now,
          'updated_at' => $now,
        ])
        ->execute() !== NULL;
    }
  }

  /**
   * Build initial messages array with system prompt and Q&A pairs.
   *
   * @return array
   *   Messages array.
   */
  public function buildInitialMessages() {
    $settings = $this->settingsManager->getSettings();
    $messages = [];

    // Enhanced system prompt with link instructions and contact collection
    $system_prompt = $settings['system_prompt'] ?? '';
    if (!empty($system_prompt)) {
      $system_prompt .= "\n\nIMPORTANT INSTRUCTIONS:\n";
      $system_prompt .= "1. When your answer references information from a specific page, always include the page URL at the end of your response if available in the Q&A context. Format links as: 'For more information, visit: [URL]'\n";
      $system_prompt .= "2. When a user asks about booking, appointments, consultations, or wants to get in touch, naturally suggest they can provide their contact information. You can say something like: 'I'd be happy to help you with that! Would you like to provide your contact information so we can get in touch with you?' or 'If you'd like, I can help you get in touch with us. Just let me know if you'd like to share your contact details.'\n";
      $system_prompt .= "3. Be conversational and helpful. Don't be pushy about collecting information, but make it easy for users to provide their details when they're interested.";
      $messages[] = [
        'role' => 'system',
        'content' => $system_prompt,
      ];
    }

    // Add Q&A pairs as user/assistant messages (these are used as context, not displayed)
    $qa_pairs = $settings['qa_pairs'] ?? [];
    foreach ($qa_pairs as $pair) {
      if (!empty($pair['question']) && !empty($pair['answer'])) {
        // Ensure URL is included in answer if available
        $answer = $pair['answer'];
        if (!empty($pair['url']) && strpos($answer, $pair['url']) === FALSE) {
          $answer .= "\n\nFor more information, visit: " . $pair['url'];
        }

        $messages[] = [
          'role' => 'user',
          'content' => $pair['question'],
        ];
        $messages[] = [
          'role' => 'assistant',
          'content' => $answer,
        ];
      }
    }

    return $messages;
  }

  /**
   * Clear conversation.
   *
   * @param string $session_id
   *   Session ID.
   *
   * @return bool
   *   TRUE on success, FALSE otherwise.
   */
  public function clearConversation($session_id) {
    return $this->database->delete('knova_conversations')
      ->condition('session_id', $session_id)
      ->execute() !== NULL;
  }

}

