<?php

namespace Drupal\knova\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Cache\CacheBackendInterface;

/**
 * Settings Manager service for Knova.
 */
class SettingsManager {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * Constructs a SettingsManager object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(ConfigFactoryInterface $config_factory, CacheBackendInterface $cache) {
    $this->configFactory = $config_factory;
    $this->cache = $cache;
  }

  /**
   * Get all settings.
   *
   * @param bool $force_refresh
   *   If TRUE, bypass cache and get fresh data from config.
   *
   * @return array
   *   Array of settings.
   */
  public function getSettings($force_refresh = FALSE) {
    $cid = 'knova:settings';
    
    // Try cache first (unless forcing refresh)
    if (!$force_refresh && ($cached = $this->cache->get($cid))) {
      return $cached->data;
    }

    $config = $this->configFactory->get('knova.settings');
    $settings = $config->get();

    // Cache for 60 seconds with proper cache tags
    $this->cache->set($cid, $settings, time() + 60, ['config:knova.settings']);

    return $settings;
  }
  
  /**
   * Clear the settings cache.
   */
  public function clearCache() {
    $this->cache->delete('knova:settings');
  }

  /**
   * Get a specific setting.
   *
   * @param string $key
   *   Setting key.
   * @param mixed $default
   *   Default value if not found.
   *
   * @return mixed
   *   Setting value.
   */
  public function getSetting($key, $default = NULL) {
    $settings = $this->getSettings();
    return $settings[$key] ?? $default;
  }

  /**
   * Update settings.
   *
   * @param array $settings
   *   Settings to update.
   */
  public function updateSettings(array $settings) {
    $config = $this->configFactory->getEditable('knova.settings');
    
    foreach ($settings as $key => $value) {
      $config->set($key, $value);
    }
    
    $config->save();
    
    // Clear cache
    $this->cache->delete('knova:settings');
  }

  /**
   * Get default settings.
   *
   * @return array
   *   Default settings.
   */
  public function getDefaults() {
    return [
      'enabled' => TRUE,
      'position' => 'right',
      'distance_bottom' => 20,
      'distance_side' => 20,
      'mobile_enabled' => TRUE,
      'primary_color' => '#0073aa',
      'secondary_color' => '#005177',
      'text_color' => '#333333',
      'accent_color' => '#0073aa',
      'widget_background_color' => '#ffffff',
      'user_message_bg_color' => '#0073aa',
      'assistant_message_bg_color' => '#ffffff',
      'border_radius' => 12,
      'widget_width_desktop' => 380,
      'widget_width_mobile' => 320,
      'widget_height_max' => 400,
      'logo_url' => '',
      'title_text' => 'Knova - Intelligence',
      'tagline_text' => 'AI-Powered Assistant',
      'welcome_message' => 'Hello! I\'m Knova, your intelligent AI assistant. How can I help you today?',
      'input_placeholder' => 'Type your message...',
      'font_family' => 'inherit',
      'model' => 'gpt-3.5-turbo',
      'temperature' => 0.7,
      'max_tokens' => 256,
      'top_p' => 1,
      'frequency_penalty' => 0,
      'presence_penalty' => 0,
      'endpoint' => 'https://api.openai.com/v1/chat/completions',
      'api_key' => '',
      'system_prompt' => 'You are a helpful AI assistant for this website. You can ONLY answer questions based on the information provided in the Q&A pairs and conversation context. If you do not have information to answer a question, politely say so and suggest the user contact support. When your answer references a specific page or topic, always include the relevant page URL at the end of your response if available. Never make up information or provide answers outside the provided context.',
      'qa_pairs' => [],
      'conversation_storage' => 'localStorage',
      'rate_limit_enabled' => TRUE,
      'rate_limit_requests' => 10,
      'rate_limit_period' => 60,
      'error_message' => 'Sorry, something went wrong. Please try again.',
      'debug_mode' => FALSE,
      'custom_css' => '',
    ];
  }

}

