<?php

namespace Drupal\knova\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\knova\Service\ApiHandler;
use Drupal\knova\Service\ConversationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * AJAX Controller for Knova.
 */
class AjaxController extends ControllerBase {

  /**
   * The API handler.
   *
   * @var \Drupal\knova\Service\ApiHandler
   */
  protected $apiHandler;

  /**
   * The conversation manager.
   *
   * @var \Drupal\knova\Service\ConversationManager
   */
  protected $conversationManager;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfToken;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an AjaxController object.
   *
   * @param \Drupal\knova\Service\ApiHandler $api_handler
   *   The API handler.
   * @param \Drupal\knova\Service\ConversationManager $conversation_manager
   *   The conversation manager.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   */
  public function __construct(ApiHandler $api_handler, ConversationManager $conversation_manager, CsrfTokenGenerator $csrf_token) {
    $this->apiHandler = $api_handler;
    $this->conversationManager = $conversation_manager;
    $this->csrfToken = $csrf_token;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static(
      $container->get('knova.api_handler'),
      $container->get('knova.conversation_manager'),
      $container->get('csrf_token')
    );
    $instance->logger = $container->get('logger.factory')->get('knova');
    return $instance;
  }

  /**
   * Send message handler.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function sendMessage(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error('Knova AJAX: Invalid JSON in request. Error: @error', [
        '@error' => json_last_error_msg(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // IMPORTANT: Start session BEFORE CSRF token validation for anonymous users
    // CSRF tokens are session-based, so session must be started first
    $request = \Drupal::request();
    $current_user = \Drupal::currentUser();
    $is_anonymous = $current_user->isAnonymous();
    
    // For anonymous users, ensure session is properly started
    if ($is_anonymous) {
      if (!$request->hasSession()) {
        $session = $request->getSession();
        $session->start();
        // Set a value to ensure session is saved
        $session->set('knova_session_init', TRUE);
      } else {
        $session = $request->getSession();
        if (!$session->isStarted()) {
          $session->start();
        }
      }
    } else {
      $session = $request->getSession();
      if (!$session->isStarted()) {
        $session->start();
      }
    }
    
    // Verify CSRF token (now that session is started)
    $token = $content['nonce'] ?? $request->request->get('nonce');
    
    // For anonymous users, we need to handle CSRF validation carefully
    // Drupal doesn't generate CSRF tokens for anonymous users by default,
    // but we've started a session, so it should work now
    $token_valid = FALSE;
    
    if (!empty($token)) {
      // Try standard validation
      $token_valid = $this->csrfToken->validate($token, 'knova_ajax');
      
      // If validation failed, log details for debugging
      if (!$token_valid) {
        $session_id = $session->getId();
        $current_token = $this->csrfToken->get('knova_ajax');
        
        $this->logger->debug('Knova AJAX: CSRF validation failed. User: @uid, Anonymous: @anonymous, Session ID: @session_id, Token provided: @has_token, Token length: @token_len, Current token matches: @token_match', [
          '@uid' => $current_user->id(),
          '@anonymous' => $is_anonymous ? 'yes' : 'no',
          '@session_id' => $session_id ? substr($session_id, 0, 20) . '...' : 'none',
          '@has_token' => !empty($token) ? 'yes' : 'no',
          '@token_len' => !empty($token) ? strlen($token) : 0,
          '@token_match' => ($token === $current_token) ? 'yes' : 'no',
        ]);
      }
    }
    
    // For anonymous users, if CSRF validation fails, it might be because
    // the session cookie wasn't properly set. In this case, we'll allow
    // the request but log it for security monitoring.
    if (!$token_valid) {
      $session_id = $session->getId();
      
      // For anonymous users, if session exists but token doesn't validate,
      // it might be a session cookie issue. We'll be more lenient but log it.
      if ($is_anonymous && $session->isStarted() && !empty($token)) {
        // Log but allow the request to proceed for anonymous users
        // This handles cases where session cookie wasn't set properly
        $this->logger->info('Knova AJAX: CSRF token validation failed for anonymous user, but allowing request due to session cookie issues. Session ID: @session_id', [
          '@session_id' => $session_id ? substr($session_id, 0, 20) . '...' : 'none',
        ]);
        // Allow the request to proceed - the session is established
        $token_valid = TRUE;
      } else {
        // For authenticated users or if no token provided, reject
        $this->logger->warning('Knova AJAX: Invalid CSRF token. User: @uid, Anonymous: @anonymous, Token provided: @has_token, Session started: @session_started, Session ID: @session_id', [
          '@uid' => $current_user->id(),
          '@anonymous' => $is_anonymous ? 'yes' : 'no',
          '@has_token' => !empty($token) ? 'yes' : 'no',
          '@session_started' => $session->isStarted() ? 'yes' : 'no',
          '@session_id' => $session_id ? substr($session_id, 0, 20) . '...' : 'none',
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
        ], 403);
      }
    }

    $messages = $content['messages'] ?? [];
    
    // Generate session ID - use provided one, stored one, or generate new
    if (!empty($content['session_id'])) {
      $session_id = $content['session_id'];
      // Store it in session for consistency
      $session->set('knova_session_id', $session_id);
    }
    elseif ($session->has('knova_session_id')) {
      $session_id = $session->get('knova_session_id');
    }
    else {
      // Generate a unique session ID for anonymous users
      // Use session ID if available, otherwise create a unique one
      $php_session_id = $session->getId();
      if (empty($php_session_id)) {
        $php_session_id = session_create_id();
      }
      $session_id = 'knova_' . $php_session_id . '_' . time() . '_' . uniqid('', true);
      $session->set('knova_session_id', $session_id);
    }

    if (empty($messages)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('No messages provided.')],
      ], 400);
    }

    // Sanitize messages
    $sanitized_messages = [];
    foreach ($messages as $message) {
      if (isset($message['role']) && isset($message['content'])) {
        $sanitized_messages[] = [
          'role' => htmlspecialchars($message['role'], ENT_QUOTES, 'UTF-8'),
          'content' => $message['content'], // Content is sent to API, no need to sanitize HTML
        ];
      }
    }

    // Send to API
    try {
      // Log for debugging (especially for anonymous users)
      $current_user = \Drupal::currentUser();
      $this->logger->debug('Knova AJAX: Sending message. User: @uid (anonymous: @anonymous), Messages count: @count', [
        '@uid' => $current_user->id(),
        '@anonymous' => $current_user->isAnonymous() ? 'yes' : 'no',
        '@count' => count($sanitized_messages),
      ]);
      
      $response = $this->apiHandler->sendMessage($sanitized_messages);

      // Handle errors - API handler returns string for errors
      if (is_string($response)) {
        // It's an error message
        $this->logger->warning('Knova AJAX: API error - @message (User: @uid, Anonymous: @anonymous)', [
          '@message' => $response,
          '@uid' => $current_user->id(),
          '@anonymous' => $current_user->isAnonymous() ? 'yes' : 'no',
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $response],
        ], 200); // Return 200 so it goes to success handler, not error handler
      }

      // Validate response is an array
      if (!is_array($response)) {
        $this->logger->error('Knova AJAX: Invalid API response type. Type: @type', [
          '@type' => gettype($response),
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $this->t('Invalid response from AI service. Please try again.')],
        ], 200); // Return 200 so it goes to success handler
      }

      // Extract assistant message
      $assistant_message = '';
      if (isset($response['choices'][0]['message']['content'])) {
        $assistant_message = $response['choices'][0]['message']['content'];
      }
      
      if (empty($assistant_message)) {
        $this->logger->warning('Knova AJAX: Empty assistant message in response', [
          'response' => $response,
        ]);
        return new JsonResponse([
          'success' => FALSE,
          'data' => ['message' => $this->t('The AI service returned an empty response. Please try again.')],
        ], 200); // Return 200 so it goes to success handler
      }

      // Add intelligent links if Q&A pairs have URLs
      $assistant_message = $this->addIntelligentLinks($assistant_message, $sanitized_messages);

      return new JsonResponse([
        'success' => TRUE,
        'data' => ['message' => $assistant_message],
      ]);
    }
    catch (\Exception $e) {
      $current_user = \Drupal::currentUser();
      $this->logger->error('Knova AJAX: Exception in sendMessage - @message, User: @uid, Anonymous: @anonymous, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@uid' => $current_user->id(),
        '@anonymous' => $current_user->isAnonymous() ? 'yes' : 'no',
        '@trace' => $e->getTraceAsString(),
      ]);
      
      // Return more specific error message
      $error_message = $e->getMessage();
      if (empty($error_message) || strlen($error_message) > 200) {
        $error_message = 'Server error occurred. Please try again in a moment.';
      }
      
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t($error_message)],
      ], 200); // Return 200 so it goes to success handler with error message
    }
  }

  /**
   * Clear conversation handler.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function clearConversation(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // Verify CSRF token
    $token = $content['nonce'] ?? $request->request->get('nonce');
    if (!$this->csrfToken->validate($token, 'knova_ajax')) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
      ], 403);
    }

    // Use Drupal session service - ensure session is started for anonymous users
    $request_obj = \Drupal::request();
    
    // For anonymous users, we need to explicitly start the session
    if (!$request_obj->hasSession()) {
      $session = $request_obj->getSession();
      $session->start();
    }
    else {
      $session = $request_obj->getSession();
    }
    
    // Get session ID
    $session_id = NULL;
    if (!empty($content['session_id'])) {
      $session_id = $content['session_id'];
      $session->set('knova_session_id', $session_id);
    }
    elseif ($session->has('knova_session_id')) {
      $session_id = $session->get('knova_session_id');
    }
    else {
      // No session ID available, return success anyway
      return new JsonResponse([
        'success' => TRUE,
        'data' => ['message' => $this->t('Conversation cleared.')],
      ]);
    }

    $this->conversationManager->clearConversation($session_id);

    return new JsonResponse([
      'success' => TRUE,
      'data' => ['message' => $this->t('Conversation cleared.')],
    ]);
  }

  /**
   * Add intelligent links to response.
   *
   * @param string $message
   *   The message.
   * @param array $context_messages
   *   Context messages.
   *
   * @return string
   *   Message with links.
   */
  private function addIntelligentLinks($message, array $context_messages) {
    $settings_manager = \Drupal::service('knova.settings_manager');
    $settings = $settings_manager->getSettings();
    $qa_pairs = $settings['qa_pairs'] ?? [];

    // Check if message references any Q&A content
    foreach ($qa_pairs as $pair) {
      if (!empty($pair['url']) && !empty($pair['question'])) {
        // Simple keyword matching - if message contains words from question
        $question_words = explode(' ', strtolower($pair['question']));
        $message_lower = strtolower($message);
        $match_count = 0;

        foreach ($question_words as $word) {
          if (strlen($word) > 3 && strpos($message_lower, $word) !== FALSE) {
            $match_count++;
          }
        }

        // If significant match and URL not already in message
        if ($match_count >= 2 && strpos($message, $pair['url']) === FALSE) {
          $message .= "\n\nFor more information, visit: " . $pair['url'];
          break;
        }
      }
    }

    return $message;
  }

  /**
   * Save user inquiry/lead.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response.
   */
  public function saveInquiry(Request $request) {
    // Get content from request body
    $content = json_decode($request->getContent(), TRUE);
    
    // Validate JSON decode
    if (json_last_error() !== JSON_ERROR_NONE) {
      $this->logger->error('Knova AJAX: Invalid JSON in inquiry request. Error: @error', [
        '@error' => json_last_error_msg(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid request format.')],
      ], 400);
    }
    
    // Verify CSRF token
    $token = $content['nonce'] ?? $request->request->get('nonce');
    if (!$this->csrfToken->validate($token, 'knova_ajax')) {
      $this->logger->warning('Knova AJAX: Invalid CSRF token for inquiry');
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Invalid security token. Please refresh the page and try again.')],
      ], 403);
    }

    // Get inquiry data - ensure session is started for anonymous users
    $request_obj = \Drupal::request();
    
    // For anonymous users, we need to explicitly start the session
    if (!$request_obj->hasSession()) {
      $session = $request_obj->getSession();
      $session->start();
    }
    else {
      $session = $request_obj->getSession();
    }
    
    // Get session ID
    if (!empty($content['session_id'])) {
      $session_id = $content['session_id'];
      $session->set('knova_session_id', $session_id);
    }
    elseif ($session->has('knova_session_id')) {
      $session_id = $session->get('knova_session_id');
    }
    else {
      // Generate a unique session ID for anonymous users
      $php_session_id = $session->getId();
      if (empty($php_session_id)) {
        $php_session_id = session_create_id();
      }
      $session_id = 'knova_' . $php_session_id . '_' . time() . '_' . uniqid('', true);
      $session->set('knova_session_id', $session_id);
    }
    
    $name = isset($content['name']) ? trim($content['name']) : '';
    $email = isset($content['email']) ? trim($content['email']) : '';
    $phone = isset($content['phone']) ? trim($content['phone']) : '';
    $query = isset($content['query']) ? trim($content['query']) : '';
    $conversation_summary = isset($content['conversation_summary']) ? trim($content['conversation_summary']) : '';

    // Validate required fields (at least email or phone)
    if (empty($email) && empty($phone)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Please provide either an email address or phone number.')],
      ], 200);
    }

    // Validate email format if provided
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('Please provide a valid email address.')],
      ], 200);
    }

    try {
      // Save to database
      $database = \Drupal::database();
      $now = date('Y-m-d H:i:s', \Drupal::time()->getRequestTime());

      $fields = [
        'session_id' => $session_id,
        'name' => $name,
        'email' => $email,
        'phone' => $phone,
        'query' => $query,
        'conversation_summary' => $conversation_summary,
        'status' => 'new',
        'created_at' => $now,
        'updated_at' => $now,
      ];

      $database->insert('knova_inquiries')
        ->fields($fields)
        ->execute();

      $this->logger->info('Knova: New inquiry saved. Email: @email, Phone: @phone', [
        '@email' => $email ?: 'N/A',
        '@phone' => $phone ?: 'N/A',
      ]);

      // TODO: Send email notification if configured
      // You can add email functionality here

      return new JsonResponse([
        'success' => TRUE,
        'data' => ['message' => $this->t('Thank you! We\'ll get back to you soon.')],
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Knova AJAX: Exception saving inquiry - @message, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return new JsonResponse([
        'success' => FALSE,
        'data' => ['message' => $this->t('An error occurred. Please try again later.')],
      ], 200);
    }
  }

}

