<?php

namespace Drupal\knova\Service;

use Drupal\Core\Cache\CacheBackendInterface;
use Psr\Log\LoggerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * API Handler service for Knova.
 */
class ApiHandler {

  /**
   * The settings manager.
   *
   * @var \Drupal\knova\Service\SettingsManager
   */
  protected $settingsManager;

  /**
   * The cache backend.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cache;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs an ApiHandler object.
   *
   * @param \Drupal\knova\Service\SettingsManager $settings_manager
   *   The settings manager.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(SettingsManager $settings_manager, CacheBackendInterface $cache, LoggerInterface $logger, ClientInterface $http_client) {
    $this->settingsManager = $settings_manager;
    $this->cache = $cache;
    $this->logger = $logger;
    $this->httpClient = $http_client;
  }

  /**
   * Send message to API.
   *
   * @param array $messages
   *   Messages array.
   *
   * @return array|string
   *   Response array or error message string.
   */
  public function sendMessage(array $messages) {
    // Force refresh settings to get latest API key (bypass cache)
    $settings = $this->settingsManager->getSettings(TRUE);
    
    // Log for debugging (especially for anonymous users)
    $current_user = \Drupal::currentUser();
    $this->logger->debug('Knova API: Getting settings. User: @uid (anonymous: @anonymous), Has endpoint: @endpoint, Has API key: @apikey', [
      '@uid' => $current_user->id(),
      '@anonymous' => $current_user->isAnonymous() ? 'yes' : 'no',
      '@endpoint' => !empty($settings['endpoint']) ? 'yes' : 'no',
      '@apikey' => !empty($settings['api_key']) ? 'yes' : 'no',
    ]);

    // Validate endpoint
    if (empty($settings['endpoint'])) {
      $this->logger->error('Knova API: Endpoint is empty. User: @uid', ['@uid' => $current_user->id()]);
      return 'API endpoint URL is not configured. Please check your settings.';
    }

    if (!filter_var($settings['endpoint'], FILTER_VALIDATE_URL)) {
      $this->logger->error('Knova API: Invalid endpoint URL: @endpoint. User: @uid', [
        '@endpoint' => $settings['endpoint'],
        '@uid' => $current_user->id(),
      ]);
      return 'Invalid API endpoint URL. Please check your settings.';
    }

    // Validate API key
    if (empty($settings['api_key'])) {
      $this->logger->error('Knova API: API key is empty. User: @uid', ['@uid' => $current_user->id()]);
      return 'API key is not configured. Please add your API key in the module settings.';
    }

    // Check rate limiting
    if (!$this->checkRateLimit()) {
      $rate_limit_requests = $settings['rate_limit_requests'] ?? 10;
      $rate_limit_period = $settings['rate_limit_period'] ?? 60;
      return sprintf('You\'ve reached the maximum number of responses (%d requests per %d seconds). Please wait a moment and try again.', $rate_limit_requests, $rate_limit_period);
    }

    $body = [
      'model' => $settings['model'] ?? 'gpt-3.5-turbo',
      'messages' => $messages,
      'temperature' => (float) ($settings['temperature'] ?? 0.7),
      'max_tokens' => (int) ($settings['max_tokens'] ?? 256),
      'top_p' => (float) ($settings['top_p'] ?? 1),
      'frequency_penalty' => (float) ($settings['frequency_penalty'] ?? 0),
      'presence_penalty' => (float) ($settings['presence_penalty'] ?? 0),
    ];

    try {
      // Log request details (without sensitive data)
      $this->logger->debug('Knova API: Request body (without messages): @body', [
        '@body' => json_encode([
          'model' => $body['model'],
          'temperature' => $body['temperature'],
          'max_tokens' => $body['max_tokens'],
          'messages_count' => count($messages),
        ]),
      ]);
      
      // Trim API key to remove any whitespace and validate
      $api_key = trim($settings['api_key']);
      
      // Log API key info for debugging (first 10 and last 4 chars only for security)
      $api_key_length = strlen($api_key);
      $api_key_preview = $api_key_length > 14 
        ? substr($api_key, 0, 10) . '...' . substr($api_key, -4)
        : str_repeat('*', $api_key_length);
      $this->logger->info('Knova API: API key length: @length, preview: @preview', [
        '@length' => $api_key_length,
        '@preview' => $api_key_preview,
      ]);
      
      // Validate API key is not empty
      if (empty($api_key)) {
        $this->logger->error('Knova API: API key is empty after trimming.');
        return 'API key is not configured. Please add your API key in the module settings.';
      }
      
      // Validate API key format (OpenAI keys start with sk- and can contain letters, numbers, underscores, and hyphens)
      if (!preg_match('/^sk-[a-zA-Z0-9_-]+/', $api_key)) {
        $this->logger->error('Knova API: Invalid API key format. Key does not match expected pattern. Length: @length, Preview: @preview', [
          '@length' => $api_key_length,
          '@preview' => $api_key_preview,
        ]);
        return 'Invalid API key format. OpenAI API keys should start with "sk-". Please check your API key in the module settings.';
      }
      
      // Warn if API key seems too short (OpenAI keys are typically 50+ characters)
      if ($api_key_length < 40) {
        $this->logger->warning('Knova API: API key seems unusually short. Length: @length', [
          '@length' => $api_key_length,
        ]);
      }
      
      // Log request attempt (without exposing full key)
      $this->logger->debug('Knova API: Making request to @endpoint', [
        '@endpoint' => $settings['endpoint'],
      ]);
      
      $response = $this->httpClient->request('POST', $settings['endpoint'], [
        'timeout' => 30,
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => 'Bearer ' . $api_key,
        ],
        'json' => $body,
      ]);

      $response_code = $response->getStatusCode();
      $response_body_content = $response->getBody()->getContents();
      
      // Log response for debugging (without sensitive data)
      if ($settings['debug_mode'] ?? FALSE) {
        $this->logger->debug('Knova API response: Status @code, Body length: @length', [
          '@code' => $response_code,
          '@length' => strlen($response_body_content),
        ]);
      }
      
      $response_body = json_decode($response_body_content, TRUE);
      
      // Check for JSON decode errors
      if (json_last_error() !== JSON_ERROR_NONE) {
        $this->logger->error('Knova API: Failed to decode JSON response. Error: @error, Response: @response', [
          '@error' => json_last_error_msg(),
          '@response' => substr($response_body_content, 0, 500), // Log first 500 chars
        ]);
        return 'Invalid response from AI service. Please check your API configuration.';
      }

      if ($response_code !== 200) {
        return $this->handleHttpError($response_code, $response_body ?? []);
      }
      
      // Validate response structure
      if (!isset($response_body['choices']) || !is_array($response_body['choices']) || empty($response_body['choices'])) {
        $this->logger->error('Knova API: Invalid response structure. Response: @response', [
          '@response' => substr($response_body_content, 0, 500),
        ]);
        return 'The AI service returned an invalid response. Please check your API configuration.';
      }

      // Update rate limit
      $this->updateRateLimit();

      return $response_body;
    }
    catch (RequestException $e) {
      $response = $e->hasResponse() ? $e->getResponse() : NULL;
      $status_code = $response ? $response->getStatusCode() : 0;
      $response_body = $response ? $response->getBody()->getContents() : '';
      
      // Log the API key info again for 401/403 errors to help debug
      if ($status_code == 401 || $status_code == 403) {
        $api_key_length = strlen(trim($settings['api_key'] ?? ''));
        $api_key_preview = $api_key_length > 14 
          ? substr(trim($settings['api_key'] ?? ''), 0, 10) . '...' . substr(trim($settings['api_key'] ?? ''), -4)
          : str_repeat('*', $api_key_length);
        $this->logger->error('Knova API Authentication Error: Status @status, API key length: @length, preview: @preview, Response: @response', [
          '@status' => $status_code,
          '@length' => $api_key_length,
          '@preview' => $api_key_preview,
          '@response' => substr($response_body, 0, 500),
        ]);
      } else {
        $this->logger->error('Knova API RequestException: @message, Status: @status, Response: @response', [
          '@message' => $e->getMessage(),
          '@status' => $status_code,
          '@response' => substr($response_body, 0, 500),
        ]);
      }
      
      // If we have a response, try to handle it as HTTP error
      if ($response && $status_code > 0) {
        $response_data = json_decode($response_body, TRUE);
        return $this->handleHttpError($status_code, $response_data ?? []);
      }
      
      return $this->handleConnectionError($e);
    }
    catch (\Exception $e) {
      $this->logger->error('Knova API error: @message, File: @file, Line: @line, Trace: @trace', [
        '@message' => $e->getMessage(),
        '@file' => $e->getFile(),
        '@line' => $e->getLine(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return 'Unable to connect to the AI service. Error: ' . $e->getMessage();
    }
  }

  /**
   * Handle HTTP errors.
   *
   * @param int $response_code
   *   HTTP response code.
   * @param array $response_body
   *   Response body.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   Error message.
   */
  private function handleHttpError($response_code, array $response_body) {
    $error_message = $response_body['error']['message'] ?? '';

    switch ($response_code) {
      case 401:
      case 403:
        return 'Authentication failed. Please check your API key in the module settings.';

      case 404:
        return 'The AI service endpoint was not found. Please check your API endpoint URL in the module settings.';

      case 429:
        return 'Too many requests. Please wait a moment and try again.';

      case 500:
      case 502:
      case 503:
      case 504:
        return 'The AI service is temporarily unavailable. Please try again in a few moments.';

      default:
        if (empty($error_message)) {
          return sprintf('The AI service returned an error (Status: %d). Please try again later.', $response_code);
        }
        return $error_message;
    }
  }

  /**
   * Handle connection errors.
   *
   * @param \GuzzleHttp\Exception\RequestException $e
   *   The exception.
   *
   * @return string
   *   Error message.
   */
  private function handleConnectionError(RequestException $e) {
    $message = $e->getMessage();

    if (strpos(strtolower($message), 'timeout') !== FALSE) {
      return 'The request timed out. The AI service may be slow or unavailable. Please try again in a moment.';
    }

    if (strpos(strtolower($message), 'connection') !== FALSE || strpos(strtolower($message), 'network') !== FALSE) {
      return 'Unable to connect to the AI service. Please check your internet connection and try again.';
    }

    return 'Unable to connect to the AI service. Please try again later.';
  }

  /**
   * Check rate limit.
   *
   * @return bool
   *   TRUE if under limit, FALSE otherwise.
   */
  private function checkRateLimit() {
    $settings = $this->settingsManager->getSettings();

    if (!($settings['rate_limit_enabled'] ?? TRUE)) {
      return TRUE;
    }

    $ip = $this->getClientIp();
    $cid = 'knova_rate_limit:' . md5($ip);
    $cache_item = $this->cache->get($cid);

    if (!$cache_item) {
      // Initialize on first request
      $this->cache->set($cid, 0, time() + ($settings['rate_limit_period'] ?? 60));
      return TRUE;
    }

    $requests = $cache_item->data;
    return $requests < ($settings['rate_limit_requests'] ?? 10);
  }

  /**
   * Update rate limit.
   */
  private function updateRateLimit() {
    $settings = $this->settingsManager->getSettings();
    $ip = $this->getClientIp();
    $cid = 'knova_rate_limit:' . md5($ip);
    $cache_item = $this->cache->get($cid);

    $requests = $cache_item ? $cache_item->data : 0;
    $this->cache->set($cid, $requests + 1, time() + ($settings['rate_limit_period'] ?? 60));
  }

  /**
   * Get client IP address.
   *
   * @return string
   *   Client IP address.
   */
  private function getClientIp() {
    $ip_keys = [
      'HTTP_CLIENT_IP',
      'HTTP_X_FORWARDED_FOR',
      'HTTP_X_FORWARDED',
      'HTTP_X_CLUSTER_CLIENT_IP',
      'HTTP_FORWARDED_FOR',
      'HTTP_FORWARDED',
      'REMOTE_ADDR',
    ];

    $request = \Drupal::request();
    foreach ($ip_keys as $key) {
      if ($request->server->has($key)) {
        $ips = explode(',', $request->server->get($key));
        foreach ($ips as $ip) {
          $ip = trim($ip);
          if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) !== FALSE) {
            return $ip;
          }
        }
      }
    }

    return $request->getClientIp() ?? '0.0.0.0';
  }

}

