<?php

namespace Drupal\knova\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Database\Query\TableSortExtender;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for managing inquiries.
 */
class InquiriesController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs an InquiriesController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(Connection $database, DateFormatterInterface $date_formatter) {
    $this->database = $database;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays a listing of inquiries.
   *
   * @return array
   *   A render array as expected by the renderer.
   */
  public function overview() {
    $header = [
      [
        'data' => $this->t('ID'),
        'field' => 'id',
        'sort' => 'desc',
      ],
      [
        'data' => $this->t('Name'),
        'field' => 'name',
      ],
      [
        'data' => $this->t('Email'),
        'field' => 'email',
      ],
      [
        'data' => $this->t('Phone'),
        'field' => 'phone',
      ],
      [
        'data' => $this->t('Query'),
        'field' => 'query',
      ],
      [
        'data' => $this->t('Status'),
        'field' => 'status',
      ],
      [
        'data' => $this->t('Created'),
        'field' => 'created_at',
        'sort' => 'desc',
      ],
      [
        'data' => $this->t('Operations'),
      ],
    ];

    $query = $this->database->select('knova_inquiries', 'ki')
      ->extend(PagerSelectExtender::class)
      ->extend(TableSortExtender::class);

    $query->fields('ki', [
      'id',
      'session_id',
      'name',
      'email',
      'phone',
      'query',
      'conversation_summary',
      'status',
      'created_at',
      'updated_at',
    ]);

    $query->orderByHeader($header);
    $query->limit(50);

    $result = $query->execute();

    $rows = [];
    foreach ($result as $inquiry) {
      // Truncate query for display
      $query_text = $inquiry->query;
      if (mb_strlen($query_text) > 100) {
        $query_text = mb_substr($query_text, 0, 100) . '...';
      }

      // Format date
      $created_date = '';
      if ($inquiry->created_at) {
        try {
          $timestamp = strtotime($inquiry->created_at);
          if ($timestamp !== FALSE) {
            $created_date = $this->dateFormatter->format($timestamp, 'short');
          }
        }
        catch (\Exception $e) {
          $created_date = $inquiry->created_at;
        }
      }

      // Status badge
      $status_class = 'status-' . strtolower($inquiry->status);
      $status_label = ucfirst($inquiry->status);

      // Operations
      $view_url = Url::fromRoute('knova.inquiry_view', ['id' => $inquiry->id]);
      $operations = [
        '#type' => 'operations',
        '#links' => [
          'view' => [
            'title' => $this->t('View'),
            'url' => $view_url,
          ],
        ],
      ];

      $rows[] = [
        'data' => [
          $inquiry->id,
          $inquiry->name ?: $this->t('—'),
          $inquiry->email ?: $this->t('—'),
          $inquiry->phone ?: $this->t('—'),
          $query_text ?: $this->t('—'),
          [
            'data' => [
              '#markup' => '<span class="status-badge ' . $status_class . '">' . $status_label . '</span>',
            ],
          ],
          $created_date,
          ['data' => $operations],
        ],
      ];
    }

    $build['inquiries_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No inquiries found.'),
      '#attributes' => ['class' => ['knova-inquiries-table']],
    ];

    $build['pager'] = [
      '#type' => 'pager',
    ];

    $build['#attached']['library'][] = 'knova/admin';

    return $build;
  }

  /**
   * Displays a single inquiry.
   *
   * @param int $id
   *   The inquiry ID.
   *
   * @return array
   *   A render array.
   */
  public function view($id) {
    $inquiry = $this->database->select('knova_inquiries', 'ki')
      ->fields('ki')
      ->condition('id', $id)
      ->execute()
      ->fetchObject();

    if (!$inquiry) {
      return [
        '#markup' => $this->t('Inquiry not found.'),
      ];
    }

    // Format dates
    $created_date = '';
    $updated_date = '';
    if ($inquiry->created_at) {
      try {
        $timestamp = strtotime($inquiry->created_at);
        if ($timestamp !== FALSE) {
          $created_date = $this->dateFormatter->format($timestamp, 'long');
        }
      }
      catch (\Exception $e) {
        $created_date = $inquiry->created_at;
      }
    }
    if ($inquiry->updated_at) {
      try {
        $timestamp = strtotime($inquiry->updated_at);
        if ($timestamp !== FALSE) {
          $updated_date = $this->dateFormatter->format($timestamp, 'long');
        }
      }
      catch (\Exception $e) {
        $updated_date = $inquiry->updated_at;
      }
    }

    $rows = [
      [
        ['data' => $this->t('ID'), 'header' => TRUE],
        $inquiry->id,
      ],
      [
        ['data' => $this->t('Name'), 'header' => TRUE],
        $inquiry->name ?: $this->t('—'),
      ],
      [
        ['data' => $this->t('Email'), 'header' => TRUE],
        $inquiry->email ?: $this->t('—'),
      ],
      [
        ['data' => $this->t('Phone'), 'header' => TRUE],
        $inquiry->phone ?: $this->t('—'),
      ],
      [
        ['data' => $this->t('Query'), 'header' => TRUE],
        [
          '#type' => 'processed_text',
          '#text' => $inquiry->query ?: $this->t('—'),
          '#format' => 'plain_text',
        ],
      ],
      [
        ['data' => $this->t('Conversation Summary'), 'header' => TRUE],
        [
          '#type' => 'processed_text',
          '#text' => $inquiry->conversation_summary ?: $this->t('—'),
          '#format' => 'plain_text',
        ],
      ],
      [
        ['data' => $this->t('Status'), 'header' => TRUE],
        [
          '#markup' => '<span class="status-badge status-' . strtolower($inquiry->status) . '">' . ucfirst($inquiry->status) . '</span>',
        ],
      ],
      [
        ['data' => $this->t('Session ID'), 'header' => TRUE],
        $inquiry->session_id,
      ],
      [
        ['data' => $this->t('Created'), 'header' => TRUE],
        $created_date,
      ],
      [
        ['data' => $this->t('Updated'), 'header' => TRUE],
        $updated_date,
      ],
    ];

    $build['inquiry'] = [
      '#type' => 'table',
      '#rows' => $rows,
      '#attributes' => ['class' => ['knova-inquiry-detail']],
    ];

    $build['back_link'] = [
      '#type' => 'link',
      '#title' => $this->t('← Back to Inquiries'),
      '#url' => Url::fromRoute('knova.inquiries'),
      '#attributes' => ['class' => ['button']],
    ];

    return $build;
  }

}

