<?php

namespace Drupal\kordiam\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\kordiam\Collection\NoticesCollection;
use Drupal\kordiam\Controller\ModuleSettings;
use Drupal\kordiam\Controller\RequestsController;
use Drupal\kordiam\DeleteMethods;
use Drupal\kordiam\PageTemplate\PageTemplate;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring Publication custom field matching settings.
 */
class KordiamPublicationCustomFieldMatching extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The delete methods service.
   *
   * @var \Drupal\kordiam\DeleteMethods
   */
  protected $deleteMethods;

  /**
   * The requests controller.
   *
   * @var \Drupal\kordiam\Controller\RequestsController
   */
  protected $requestsController;

  /**
   * Constructs a KordiamPublicationCustomFieldMatching form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\kordiam\DeleteMethods $delete_methods
   *   The delete methods service.
   * @param \Drupal\kordiam\Controller\RequestsController $requests_controller
   *   The requests controller.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, TranslationInterface $string_translation, DeleteMethods $delete_methods, RequestsController $requests_controller) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->stringTranslation = $string_translation;
    $this->deleteMethods = $delete_methods;
    $this->requestsController = $requests_controller;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('config.factory'),
    $container->get('messenger'),
    $container->get('string_translation'),
    $container->get('kordiam.delete_methods'),
    $container->get('kordiam.requests_controller')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'kordiam_publication_custom_field_matching';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Loading all Drupal Content types from Kordiam.
    $is_publication_custom_fields_exist = $this->getPublicationCustomFields();
    $config = $this->configFactory->get('kordiam.settings');
    $configuration_setting_name = 'kordiam_selected_publication_custom_field_matching_list';
    $publication_cf_list = $config->get('kordiam_publication_custom_fields');

    if ($publication_cf_list !== NULL) {
      $publication_cf_list = unserialize($publication_cf_list, ['allowed_classes' => FALSE]);
    }

    if (!empty($config->get('platform_id'))) {
      $html = '<h2>' . $this->stringTranslation->translate('Publication Custom Field Matching') . '</h2>';
      $html .= '<p>';
      $html .= $this->stringTranslation->translate('Use this page to match the publication custom fields in Kordiam with the Title and URL alias in Drupal.');
      $html .= '</p>';

      if ($is_publication_custom_fields_exist && !empty(ModuleSettings::variableGet('kordiam_token'))) {
        $drupal_to_kordiam = [
          'title' => [
            'id' => 'title',
            'name' => $this->stringTranslation->translate('Title'),
          ],
          'url_alias' => [
            'id' => 'url_alias',
            'name' => $this->stringTranslation->translate('URL Alias'),
          ],
          'no_publication_cf' => [
            'id' => 'no_publication_cf',
            'name' => $this->stringTranslation->translate('No Sync'),
          ],
        ];

        if (!empty($publication_cf_list)) {
          foreach ($publication_cf_list as $value) {
            $kordiam_to_drupal[$value['id']]['id'] = $value['id'];
            $kordiam_to_drupal[$value['id']]['name'] = $value['name'];
          }
        }

        $html .= '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Kordiam to Drupal') . '</h4>';

        $form['html'] = [
          '#markup' => $html,
        ];

        $form['kordiam_to_drupal_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
        $configuration_setting_name, $kordiam_to_drupal, $drupal_to_kordiam,
        'publication_cf', 'kordiam_to_drupal', 'no_publication_cf'
        );

        $sub_title = '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Drupal to Kordiam') . '</h4>';

        $form['subTitle'] = [
          '#markup' => $sub_title,
        ];

        unset($drupal_to_kordiam['no_publication_cf']);
        $kordiam_to_drupal['no_publication_cf'] = [
          'id' => 'no_publication_cf',
          'name' => $this->stringTranslation->translate('No Sync'),
        ];

        $form['drupal_to_kordiam_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
        $configuration_setting_name, $drupal_to_kordiam, $kordiam_to_drupal,
        'publication_cf', 'drupal_to_kordiam', 'no_publication_cf'
        );

        $form['submit'] = [
          '#type' => 'submit',
          '#value' => $this->stringTranslation->translate('Save'),
        ];

        return $form;
      }
      elseif (!$is_publication_custom_fields_exist && empty($publication_cf_list)) {
        $html .= '<p>';
        $html .= '<strong>' . $this->stringTranslation->translate('Please note:') . '</strong> ';
        $html .= $this->stringTranslation->translate('The synchronized platform in Kordiam lacks publication custom fields for matching.');
        $html .= '</p>';

        $form['html'] = [
          '#markup' => $html,
        ];

        return $form;
      }
      else {
        $this->messenger->addError(NoticesCollection::getNotice(9));
      }
    }
    else {
      $this->messenger->addError(NoticesCollection::getNotice(10));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $formValues = NULL;

    if (!empty($form_state->getValues())) {
      foreach ($form_state->getValues() as $key => $value) {
        if ($key != 'form_id' && $key != 'op' && $key != 'form_token' &&
                    $key != 'form_build_id' && $key != 'submit'
        ) {
          $formValues[$key] = $value;
        }
      }

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_selected_publication_custom_field_matching_list', serialize($formValues))
        ->save();

      $this->messenger->addStatus(NoticesCollection::getNotice(13));
    }
  }

  /**
   * Perform get Types for Kordiam platform.
   *
   * @return bool
   *   The result loading types list from Kordiam.
   */
  private function getPublicationCustomFields() {
    $config = $this->configFactory->get('kordiam.settings');
    $configFactory = $this->configFactory->getEditable('kordiam.settings');
    $platform_id = $config->get('platform_id');
    $saved_publication_cf_list = $config->get('kordiam_publication_custom_fields');
    $drupal_sync_field_list = [
      'title',
      'url_alias',
      'no_publication_cf',
    ];

    if ($saved_publication_cf_list !== NULL) {
      $saved_publication_cf_list = unserialize($saved_publication_cf_list, ['allowed_classes' => FALSE]);
    }

    if (!empty($platform_id)) {
      $new_publication_cf_list = $this->requestsController->get(ModuleSettings::DN_BASE_URL, 'publication-custom-fields', NULL, 'v1_0_0');

      if ($new_publication_cf_list === 'unauthorized') {
        return FALSE;
      }

      if (!empty($new_publication_cf_list['message']) || $new_publication_cf_list === 'not_show_new_notice') {
        return FALSE;
      }

      $new_publication_cf_list = json_decode($new_publication_cf_list, TRUE);
      $new_publication_cf_list = array_filter($new_publication_cf_list, function ($item) {
        return isset($item['active']) && $item['active'] === TRUE && $item['type'] === 1;
      });

      if (empty($new_publication_cf_list)) {
        $configFactory->set('kordiam_publication_custom_fields', serialize($new_publication_cf_list))
          ->save();
        $configFactory->set('kordiam_selected_publication_custom_field_matching_list', serialize([]))
          ->save();
        return FALSE;
      }

      $this->deleteMethods->shapeDeletedItems(
      'kordiam_selected_publication_custom_field_matching_list',
      $new_publication_cf_list,
      $saved_publication_cf_list,
      $drupal_sync_field_list,
      'publication_cf'
      );
      $configFactory->set('kordiam_publication_custom_fields', serialize($new_publication_cf_list))->save();

      return TRUE;
    }

    return FALSE;
  }

}
