<?php

namespace Drupal\kordiam\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\kordiam\Collection\NoticesCollection;
use Drupal\kordiam\Controller\ActionController;
use Drupal\kordiam\Controller\ModuleSettings;
use Drupal\kordiam\Controller\RequestsController;
use Drupal\kordiam\DeleteMethods;
use Drupal\kordiam\PageTemplate\PageTemplate;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring Kordiam status matching settings.
 */
class KordiamStatusMatching extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The delete methods service.
   *
   * @var \Drupal\kordiam\DeleteMethods
   */
  protected $deleteMethods;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The requests controller.
   *
   * @var \Drupal\kordiam\Controller\RequestsController
   */
  protected $requestsController;

  /**
   * Constructs a KordiamStatusMatching form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\kordiam\DeleteMethods $delete_methods
   *   The delete methods service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\kordiam\Controller\RequestsController $requests_controller
   *   The requests controller.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, DeleteMethods $delete_methods, TranslationInterface $string_translation, RequestsController $requests_controller) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->deleteMethods = $delete_methods;
    $this->stringTranslation = $string_translation;
    $this->requestsController = $requests_controller;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('config.factory'),
    $container->get('messenger'),
    $container->get('kordiam.delete_methods'),
    $container->get('string_translation'),
    $container->get('kordiam.requests_controller')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'kordiam_status_matching';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $configuration_setting_name = 'kordiam_selected_status_matching_list';
    $loaded_kordiam_statuses = $this->getKordiamStatuses();
    $platform_id = $this->configFactory->get('kordiam.settings')->get('platform_id');

    if (!empty($platform_id)) {
      if (!empty(ModuleSettings::variableGet('kordiam_token')) && $loaded_kordiam_statuses) {
        $html = '<h2>' . $this->stringTranslation->translate('Status Matching') . '</h2>';
        $html .= $this->stringTranslation->translate('Use this page to match publication statuses in Kordiam to those in Drupal and vice versa.');

        $form['html'] = [
          '#markup' => $html,
        ];

        $initial_status_syncing = (bool) $this->configFactory->get('kordiam.settings')
          ->get('initial_status_syncing') ?? FALSE;

        $form['initial_status_syncing'] = [
          '#type' => 'checkbox',
          '#title' => $this->stringTranslation->translate('Only allow initial status syncing'),
          '#default_value' => $initial_status_syncing,
          '#required' => FALSE,
        ];

        $sub_title_dn_to_drupal = '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Kordiam to Drupal') . '</h4>';

        $form['subTitleDNToDrupal'] = [
          '#markup' => $sub_title_dn_to_drupal,
        ];

        $kordiam_to_drupal = $loaded_kordiam_statuses['activated'];
        $kordiam_to_drupal['kordiam_removed']['id'] = 'kordiam_removed';
        $kordiam_to_drupal['kordiam_removed']['name'] = $this->stringTranslation->translate('Deleted/removed');

        $drupal_to_kordiam = [
        [
          'id' => 0,
          'name' => $this->stringTranslation->translate('Unpublished'),
        ],
        [
          'id' => 1,
          'name' => $this->stringTranslation->translate('Published'),
        ],
        ];

        $field_name = ActionController::generateFieldMatchingName('status', 'kordiam_to_drupal', '5');

        $matching_value = ActionController::getFieldMatchingValue($configuration_setting_name, $field_name);

        if (empty($matching_value) && $matching_value != 0) {
          ActionController::setFieldMatchingValue($configuration_setting_name, $field_name, '1');
        }

        $form['kordiam_to_drupal_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
        $configuration_setting_name, $kordiam_to_drupal, $drupal_to_kordiam,
        'status', 'kordiam_to_drupal', 0
        );

        unset($kordiam_to_drupal['kordiam_removed']);

        if (!empty($loaded_kordiam_statuses['deactivated'])) {
          foreach ($loaded_kordiam_statuses['deactivated'] as $value) {
            $kordiam_to_drupal[] = $value;
          }
        }

        $sub_title_drupal_to_dn = '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Drupal to Kordiam') . '</h4>';

        $form['subTitleDrupalToDN'] = [
          '#markup' => $sub_title_drupal_to_dn,
        ];

        $field_name = ActionController::generateFieldMatchingName('status', 'drupal_to_kordiam', '1');

        $kordiam_to_drupal['no_sync'] = [
          'id' => 'no_sync',
          'name' => $this->stringTranslation->translate('No Sync'),
        ];

        $matching_value = ActionController::getFieldMatchingValue($configuration_setting_name, $field_name);

        if (empty($matching_value) && $matching_value != 0) {
          ActionController::setFieldMatchingValue($configuration_setting_name, $field_name, '5');
        }

        $form['drupal_to_kordiam_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
        $configuration_setting_name, $drupal_to_kordiam, $kordiam_to_drupal,
        'status', 'drupal_to_kordiam', 1
        );

        $form['submit'] = [
          '#type' => 'submit',
          '#value' => $this->stringTranslation->translate('Save'),
        ];

        return $form;
      }
      else {
        $this->messenger->addError(NoticesCollection::getNotice(9));
      }
    }
    else {
      $this->messenger->addError(NoticesCollection::getNotice(10));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $formValues = NULL;

    if (!empty($form_state->getValues())) {
      foreach ($form_state->getValues() as $key => $value) {
        if ($key != 'form_id' && $key != 'op' && $key != 'form_token' &&
                    $key != 'form_build_id' && $key != 'submit') {
          $formValues[$key] = $value;
        }
      }

      $initial_status_syncing = $form_state->getValue('initial_status_syncing');

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_selected_status_matching_list', serialize($formValues))
        ->set('initial_status_syncing', $initial_status_syncing)
        ->save();

      $this->messenger->addStatus(NoticesCollection::getNotice(13));
    }
  }

  /**
   * Perform get Status list from Kordiam.
   *
   * @return bool|array
   *   The result loading status list from Kordiam.
   */
  private function getKordiamStatuses() {
    $kordiam_to_drupal = [];
    $config = $this->configFactory->get('kordiam.settings');
    $platform_id = $config->get('platform_id');

    if (!empty($platform_id)) {
      $kordiam_status_list = $this->requestsController->get(ModuleSettings::DN_BASE_URL,
      'publication-status/platform', $platform_id);

      if ($kordiam_status_list === 'unauthorized') {
        $this->messenger->addError(NoticesCollection::getNotice(9));

        return FALSE;
      }

      $kordiam_status_list = json_decode($kordiam_status_list, TRUE);

      if (!empty($kordiam_status_list['message']) || $kordiam_status_list === 'not_show_new_notice'
         || empty($kordiam_status_list)) {
        return FALSE;
      }

      $kordiam_to_drupal['activated'] = array_filter($kordiam_status_list, function ($item) {
        return isset($item['active']) && $item['active'] === TRUE;
      });
      $kordiam_to_drupal['deactivated'] = array_filter($kordiam_status_list, function ($item) {
        return isset($item['active']) && $item['active'] === FALSE;
      });

      $saved_activated_status_list = $config->get('kordiam_activate_status_list');
      $saved_status_list_in_configuration_settings = [];

      if ($saved_activated_status_list !== NULL) {
        $saved_status_list_in_configuration_settings = unserialize($saved_activated_status_list, ['allowed_classes' => FALSE]);
      }

      $new_status_list_from_kordiam = $kordiam_to_drupal['activated'];
      // Unpublished and Published statuses.
      $drupal_status_list = [0, 1];
      // Remove incorrect status matching.
      $this->deleteMethods->shapeDeletedItems('kordiam_selected_status_matching_list',
      $new_status_list_from_kordiam, $saved_status_list_in_configuration_settings, $drupal_status_list, 'status', FALSE);

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_activate_status_list', serialize($kordiam_to_drupal['activated']))
        ->set('kordiam_deactivate_status_list', serialize($kordiam_to_drupal['deactivated']))
        ->save();
    }

    if (empty($kordiam_to_drupal)) {
      $kordiam_activate_status_list = $config->get('kordiam_activate_status_list');
      $kordiam_deactivate_status_list = $config->get('kordiam_deactivate_status_list');

      if ($kordiam_activate_status_list !== NULL) {
        $kordiam_to_drupal['activated'] = unserialize($kordiam_activate_status_list, ['allowed_classes' => FALSE]);
      }

      if ($kordiam_deactivate_status_list !== NULL) {
        $kordiam_to_drupal['deactivated'] = unserialize($kordiam_deactivate_status_list, ['allowed_classes' => FALSE]);
      }
    }

    return (!empty($kordiam_to_drupal['activated'])
      || !empty($kordiam_to_drupal['deactivated'])) ? $kordiam_to_drupal : FALSE;
  }

}
