<?php

namespace Drupal\kordiam\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\kordiam\Collection\NoticesCollection;
use Drupal\kordiam\Controller\ModuleSettings;
use Drupal\kordiam\Controller\RequestsController;
use Drupal\kordiam\DeleteMethods;
use Drupal\kordiam\PageTemplate\PageTemplate;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring Kordiam category matching settings.
 */
class KordiamCategoryMatching extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The delete methods service.
   *
   * @var \Drupal\kordiam\DeleteMethods
   */
  protected $deleteMethods;

  /**
   * The requests controller.
   *
   * @var \Drupal\kordiam\Controller\RequestsController
   */
  protected $requestsController;

  /**
   * Constructs a KordiamCategoryMatching form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\kordiam\DeleteMethods $delete_methods
   *   The delete methods service.
   * @param \Drupal\kordiam\Controller\RequestsController $requests_controller
   *   The requests controller.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, EntityTypeManagerInterface $entity_type_manager, TranslationInterface $string_translation, DeleteMethods $delete_methods, RequestsController $requests_controller) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->entityTypeManager = $entity_type_manager;
    $this->stringTranslation = $string_translation;
    $this->deleteMethods = $delete_methods;
    $this->requestsController = $requests_controller;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('config.factory'),
    $container->get('messenger'),
    $container->get('entity_type.manager'),
    $container->get('string_translation'),
    $container->get('kordiam.delete_methods'),
    $container->get('kordiam.requests_controller')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'kordiam_category_matching';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $drupal_to_kordiam = [];
    $config = $this->configFactory->get('kordiam.settings');
    $configuration_setting_name = 'kordiam_selected_category_matching_list';
    $load_category_status = $this->getCategory();
    $kordiam_category_list = $config->get('kordiam_category_list');

    if ($kordiam_category_list !== NULL) {
      $kordiam_category_list = unserialize($kordiam_category_list, ['allowed_classes' => FALSE]);
    }

    if (!empty($config->get('platform_id'))) {
      if (!empty(ModuleSettings::variableGet('kordiam_token'))) {
        if ((!$load_category_status && !empty($kordiam_category_list)) || $load_category_status !== FALSE) {
          $drupal_category_list = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();

          if (!empty($drupal_category_list['tags'])) {
            $vocabulary_term_list = $this->entityTypeManager
              ->getStorage('taxonomy_term')
              ->loadTree($drupal_category_list['tags']->id());

            if (!empty($vocabulary_term_list)) {
              foreach ($vocabulary_term_list as $term) {
                $drupal_to_kordiam[$term->tid]['id'] = $term->tid;
                $drupal_to_kordiam[$term->tid]['name'] = $term->name;

                $parents_list = $term->parents;

                if ($parents_list[0] !== '0') {
                  // Not realize case for two or more parents.
                  $drupal_to_kordiam[$term->tid]['parent'] = $parents_list[0];
                }
              }
            }
          }

          // Adding default menu item 'No category'.
          $kordiam_to_drupal['no_category']['id'] = 'no_category';
          $kordiam_to_drupal['no_category']['name'] = 'No category';

          foreach ($kordiam_category_list as $value) {
            $kordiam_to_drupal[$value['id']]['id'] = $value['id'];
            $kordiam_to_drupal[$value['id']]['name'] = $value['name'];
            if (isset($value['category'])) {
              $kordiam_to_drupal[$value['id']]['parent'] = $value['category'];
            }
          }

          $html = '<h2>' . $this->stringTranslation->translate('Category Matching') . '</h2>';
          $html .= $this->stringTranslation->translate('Use this page to match categories in Kordiam to those in Drupal and vice versa.');
          $html .= '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Kordiam to Drupal') . '</h4>';

          $form['html'] = [
            '#markup' => $html,
          ];

          $drupal_to_kordiam_new_elements['no_category']['id'] = 'no_category';
          $drupal_to_kordiam_new_elements['no_category']['name'] = 'No category';
          $drupal_to_kordiam_new_elements['do_not_import']['id'] = 'do_not_import';
          $drupal_to_kordiam_new_elements['do_not_import']['name'] = 'Do not import';

          $drupal_to_kordiam = array_merge($drupal_to_kordiam_new_elements, $drupal_to_kordiam);

          $form['kordiam_to_drupal_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
          $configuration_setting_name, $kordiam_to_drupal, $drupal_to_kordiam,
          'category', 'kordiam_to_drupal', 'no_category'
          );

          unset($drupal_to_kordiam['do_not_import'], $drupal_to_kordiam['no_category']);

          $kordiam_to_drupal = array_merge($drupal_to_kordiam_new_elements, $kordiam_to_drupal);

          $sub_title = '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Drupal to Kordiam') . '</h4>';

          if (empty($drupal_to_kordiam)) {
            $sub_title .= '<p>' . $this->stringTranslation->translate('This platform does not contain any tags to which categories could be matched.') . '</p>';
            $form['subTitle'] = [
              '#markup' => $sub_title,
            ];
          }
          else {
            $form['subTitle'] = [
              '#markup' => $sub_title,
            ];
            $form['drupal_to_kordiam_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
            $configuration_setting_name, $drupal_to_kordiam, $kordiam_to_drupal,
            'category', 'drupal_to_kordiam', 'no_category'
            );
          }

          $form['submit'] = [
            '#type' => 'submit',
            '#value' => $this->stringTranslation->translate('Save'),
          ];

          return $form;
        }
        else {
          $this->messenger->addError(NoticesCollection::getNotice(9));
        }
      }
      else {
        $this->messenger->addError(NoticesCollection::getNotice(9));
      }
    }
    else {
      $this->messenger->addError(NoticesCollection::getNotice(10));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $formValues = NULL;

    if (!empty($form_state->getValues())) {
      foreach ($form_state->getValues() as $key => $value) {
        if ($key != 'form_id' && $key != 'op' && $key != 'form_token' &&
                    $key != 'form_build_id' && $key != 'submit'
        ) {
          // Save value.
          $formValues[$key] = $value;
        }
      }

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_selected_category_matching_list', serialize($formValues))
        ->save();

      $this->messenger->addStatus(NoticesCollection::getNotice(13));
    }
  }

  /**
   * Perform get Category list from Kordiam.
   *
   * @return bool|array
   *   The result loading category list from Kordiam.
   */
  private function getCategory() {
    $config = $this->configFactory->get('kordiam.settings');
    $platform_id = $config->get('platform_id');

    $save_category_list_for_platform = $config->get('kordiam_category_list');

    if ($save_category_list_for_platform !== NULL) {
      $save_category_list_for_platform = unserialize($save_category_list_for_platform, ['allowed_classes' => FALSE]);
    }

    if (!empty($platform_id)) {
      $category_list = $this->requestsController->get(ModuleSettings::DN_BASE_URL, 'categories/platform', $platform_id);

      if ($category_list === 'unauthorized') {
        $this->messenger->addError(NoticesCollection::getNotice(9));

        return FALSE;
      }

      $category_list = json_decode($category_list, TRUE);

      if (!empty($category_list['message']) || $category_list === 'not_show_new_notice'
                || empty($category_list)) {
        return FALSE;
      }

      if ($save_category_list_for_platform) {
        $element_list_id = [];
        // Loading Drupal vocabularies.
        $drupal_category_list = $this->entityTypeManager->getStorage('taxonomy_vocabulary')->loadMultiple();

        if (isset($drupal_category_list['tags']) && $drupal_category_list['tags'] != NULL) {
          $vocabulary_term_list = $this->entityTypeManager
            ->getStorage('taxonomy_term')
            ->loadTree($drupal_category_list['tags']->id());
        }
        else {
          return FALSE;
        }

        if (!empty($vocabulary_term_list)) {
          foreach ($vocabulary_term_list as $term) {
            $element_list_id[] = $term->tid;
          }
        }

        $this->deleteMethods->shapeDeletedItems('kordiam_selected_category_matching_list', $category_list, $save_category_list_for_platform, $element_list_id, 'category');
      }

      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_category_list', serialize($category_list))
        ->save();

      return $category_list;
    }

    return FALSE;
  }

}
