<?php

namespace Drupal\kordiam\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\kordiam\Collection\NoticesCollection;
use Drupal\kordiam\Controller\ActionController;
use Drupal\kordiam\Controller\ModuleSettings;
use Drupal\kordiam\Controller\RequestsController;
use Drupal\kordiam\PageTemplate\PageTemplate;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring Kordiam task matching settings.
 */
class KordiamTaskMatching extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * The requests controller.
   *
   * @var \Drupal\kordiam\Controller\RequestsController
   */
  protected $requestsController;

  /**
   * Constructs a KordiamTaskMatching form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   * @param \Drupal\kordiam\Controller\RequestsController $requests_controller
   *   The requests controller.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, TranslationInterface $string_translation, RequestsController $requests_controller) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->stringTranslation = $string_translation;
    $this->requestsController = $requests_controller;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('config.factory'),
    $container->get('messenger'),
    $container->get('string_translation'),
    $container->get('kordiam.requests_controller')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'kordiam_task_matching';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $configuration_setting_name = 'kordiam_selected_task_matching_list';
    $kordiam_task_list = $this->getTasks();

    $platform_id = $this->configFactory->get('kordiam.settings')->get('platform_id');

    if (!empty($platform_id)) {
      if ($kordiam_task_list && !empty(ModuleSettings::variableGet('kordiam_token'))) {
        $html = '<h2>' . $this->stringTranslation->translate('Tasks Matching') . '</h2>';
        $html .= '<p>';
        $html .= $this->stringTranslation->translate('Use this page to match the default task format in Kordiam on creating a Story from Drupal to Kordiam.');
        $html .= '</p>';
        $html .= '<h4 class="dn_b-title">' . $this->stringTranslation->translate('Drupal to Kordiam') . '</h4>';

        $form['html'] = [
          '#markup' => $html,
        ];

        $kordiam_to_drupal = [
          'default_task_in_kordiam' => [
            'id' => 'default_task_in_kordiam',
            'name' => 'Default Task in Kordiam',
          ],
        ];

        foreach ($kordiam_task_list as $task) {
          $drupal_to_kordiam[$task['id']]['id'] = $task['id'];
          $drupal_to_kordiam[$task['id']]['name'] = $task['name'];
        }

        $drupal_to_kordiam = $this->setDefaultValue($drupal_to_kordiam);

        $form['kordiam_to_drupal_matching'] = PageTemplate::generateKordiamMatchingPageTemplate(
        $configuration_setting_name, $kordiam_to_drupal, $drupal_to_kordiam,
        'task', '', '18');

        $form['submit'] = [
          '#type' => 'submit',
          '#value' => $this->stringTranslation->translate('Save'),
        ];

        return $form;
      }
      else {
        $this->messenger->addError(NoticesCollection::getNotice(9));
      }
    }
    else {
      $this->messenger->addError(NoticesCollection::getNotice(10));
    }
  }

  private function setDefaultValue(array $drupal_to_kordiam = []) {
    $config = $this->configFactory->get('kordiam.settings');
    $field_name = ActionController::generateFieldMatchingName('task', '', 'default_task_in_kordiam');
    $field_value = $config->get('kordiam_selected_task_matching_list');

    if ($field_value !== NULL) {
      $field_value = unserialize($field_value, ['allowed_classes' => FALSE]);
    }

    if (!empty($field_value[$field_name]) && ($field_value[$field_name] == '18'
          || !isset($drupal_to_kordiam[$field_value[$field_name]]))) {
      $field_value[$field_name] = 18;
      $this->configFactory->getEditable('kordiam.settings')->set('kordiam_selected_task_matching_list', serialize($field_value))->save();
      $drupal_to_kordiam[18]['id'] = 18;
      $drupal_to_kordiam[18]['name'] = 'Text';
    }

    return $drupal_to_kordiam;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $formValues = NULL;

    if (!empty($form_state->getValues())) {
      foreach ($form_state->getValues() as $key => $value) {
        if ($key != 'form_id' && $key != 'op' && $key != 'form_token' &&
                    $key != 'form_build_id' && $key != 'submit'
        ) {
          $formValues[$key] = $value;
        }
      }

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_selected_task_matching_list', serialize($formValues))
        ->save();

      $this->messenger->addStatus(NoticesCollection::getNotice(13));
    }
  }

  /**
   * Retrieves tasks from the Kordiam platform.
   *
   * @return bool|array
   *   The result of loading the tasks list from Kordiam, or FALSE if
   *   unsuccessful.
   */
  private function getTasks() {
    $config = $this->configFactory->get('kordiam.settings');
    $platform_id = $config->get('platform_id');
    $saved_task_list_for_platform = $config->get('kordiam_task_list');

    if ($saved_task_list_for_platform !== NULL) {
      $saved_task_list_for_platform = unserialize($saved_task_list_for_platform, ['allowed_classes' => FALSE]);
    }

    if (!empty($platform_id)) {
      $task_list_for_platform = $this->requestsController->get(ModuleSettings::DN_BASE_URL, 'formats');

      if ($task_list_for_platform === 'unauthorized') {
        $this->messenger->addError(NoticesCollection::getNotice(9));
        return $saved_task_list_for_platform;
      }

      $task_list_for_platform = json_decode($task_list_for_platform, TRUE);

      if (!empty($task_list_for_platform['message']) || $task_list_for_platform === 'not_show_new_notice' || empty($task_list_for_platform)) {
        return $saved_task_list_for_platform;
      }

      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_task_list', serialize($task_list_for_platform))
        ->save();

      return $task_list_for_platform;
    }

    return NULL;
  }

}
