<?php

namespace Drupal\kordiam\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\Core\Url;
use Drupal\kordiam\Collection\NoticesCollection;
use Drupal\kordiam\PageTemplate\PageTemplate;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form to configure Kordiam slug synchronization settings.
 */
class KordiamSlugSyncing extends FormBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The string translation service.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * Constructs a KordiamSlugSyncing form object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $string_translation
   *   The string translation service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger, TranslationInterface $string_translation) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
    $this->stringTranslation = $string_translation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
    $container->get('config.factory'),
    $container->get('messenger'),
    $container->get('string_translation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'kordiam_slug_syncing';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $link_slug_feature = Link::fromTextAndUrl(
    $this->stringTranslation->translate('slug feature'),
    Url::fromUri("https://support.kordiam.io/hc/en-us/articles/115003545092", ['attributes' => ['target' => '_blank']])
    )->toString();

    // Slug field Kordiam.
    $kordiam_slug['slug']['id'] = 'slug_syncing';
    $kordiam_slug['slug']['name'] = $this->stringTranslation->translate('Slug (Kordiam)');

    // Set values.
    $kordiam_slug_value['kordiam_slug_permalink_val']['id'] = 'url_alias';
    $kordiam_slug_value['kordiam_slug_permalink_val']['name'] = $this->stringTranslation->translate('URL Alias');
    $kordiam_slug_value['kordiam_slug_headline_val']['id'] = 'title';
    $kordiam_slug_value['kordiam_slug_headline_val']['name'] = $this->stringTranslation->translate('Title');

    $html = '<h2>' . $this->stringTranslation->translate('Slug Syncing') . '</h2>';
    $html .= '<p>';
    $html .= $this->stringTranslation->translate('Kordiam syncs the Title, SEO Title and the URL Alias with the @link in Kordiam. Please define what field should be primarily synced to Kordiam in case of conflicts.', ['@link' => $link_slug_feature]);
    $html .= '</p>';

    $form['html'] = [
      '#markup' => $html,
    ];

    $form['kordiam_slug_syncing'] = PageTemplate::generateKordiamMatchingPageTemplate(
    'kordiam_slug_syncing', $kordiam_slug, $kordiam_slug_value, '', '', 'url_alias'
    );

    $initial_slug_syncing = (bool) $this->configFactory->get('kordiam.settings')->get('initial_slug_syncing') ?? FALSE;

    $form['initial_slug_syncing'] = [
      '#type' => 'checkbox',
      '#title' => $this->stringTranslation->translate('Only allow initial slug syncing'),
      '#default_value' => $initial_slug_syncing,
      '#required' => FALSE,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->stringTranslation->translate('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $formValues = NULL;

    if (!empty($form_state->getValues())) {
      foreach ($form_state->getValues() as $key => $value) {
        if ($key != 'form_id' && $key != 'op' && $key != 'form_token' &&
                    $key != 'form_build_id' && $key != 'submit'
        ) {
          // Save value.
          $formValues[$key] = $value;
        }
      }

      $initial_slug_syncing = $form_state->getValue('initial_slug_syncing');

      // Save the values to configuration.
      $this->configFactory->getEditable('kordiam.settings')
        ->set('kordiam_slug_syncing', serialize($formValues))
        ->set('initial_slug_syncing', $initial_slug_syncing)
        ->save();

      $this->messenger->addStatus(NoticesCollection::getNotice(13));
    }
  }

}
