<?php

namespace Drupal\kordiam\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\UrlGeneratorInterface;
use Drupal\Core\Url;

/**
 *
 */
class ContentLinkGenerator {

  protected $config;
  protected $urlGenerator;

  public function __construct(ConfigFactoryInterface $config_factory, UrlGeneratorInterface $url_generator) {
    $this->config = $config_factory->get('kordiam.settings');
    $this->urlGenerator = $url_generator;
  }

  /**
   * Generates a CMS edit link for a given node.
   *
   * This method uses Drupal's route system to generate the edit link
   * for the specified node ID. If a custom prefix is configured in
   * `url_to_content_in_cms_url_prefix`, it will be prepended to the
   * route path.
   *
   * Requires Drupal's routing and configuration systems.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return string
   *   The generated edit link URL.
   */
  public function getCmsEditLink(int $nid): string {
    $editUrl = Url::fromRoute('entity.node.edit_form', ['node' => $nid]);
    $prefix = $this->config->get('url_to_content_in_cms_url_prefix');

    if (!empty($prefix)) {
      return $prefix . $editUrl->toString();
    }

    return $editUrl->setAbsolute()->toString();
  }

  /**
   * Generates a published content link for a given node.
   *
   * If a custom prefix is defined in the configuration under
   * 'url_to_published_content_url_prefix', it will be prepended
   * to the node ID to construct the URL. Otherwise, a canonical
   * absolute URL to the node will be generated.
   *
   * Requires Drupal's routing and configuration systems.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return string
   *   The generated published content link URL.
   */
  public function getCmsOpenLink(int $nid): string {
    $prefix = $this->config->get('url_to_published_content_url_prefix');

    if (!empty($prefix)) {
      return $prefix . $nid;
    }

    $canonicalUrl = Url::fromRoute('entity.node.canonical', ['node' => $nid]);
    return $canonicalUrl->setAbsolute()->toString();
  }

  /**
   * Gets all relevant CMS links and their configured titles.
   *
   * This includes edit and open links for a given node, as well
   * as their respective titles if defined in the configuration.
   *
   * @param int $nid
   *   The node ID.
   *
   * @return array
   *   An array containing:
   *     - cmsEditLink: The edit link URL.
   *     - cmsOpenLink: The open/published link URL.
   *     - cmsEditLinkTitle (optional): The configured title for the edit link.
   *     - cmsOpenLinkTitle (optional): The configured title for the published link.
   */
  public function getLinks(int $nid): array {
    $links = [
      'cmsEditLink' => $this->getCmsEditLink($nid),
      'cmsOpenLink' => $this->getCmsOpenLink($nid),
    ];

    $url_to_content_in_cms_title = $this->config->get('url_to_content_in_cms_title');
    $url_to_published_content_title = $this->config->get('url_to_published_content_title');

    if (!empty($url_to_content_in_cms_title)) {
      $links['cmsEditLinkTitle'] = $url_to_content_in_cms_title;
    }

    if (!empty($url_to_published_content_title)) {
      $links['cmsOpenLinkTitle'] = $url_to_published_content_title;
    }

    return $links;
  }

}
