<?php

namespace Drupal\landingi_landing_pages\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\landingi_landing_pages\Service\LandingiApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\node\Entity\Node;

class LandingiAdminController extends ControllerBase {

  // UWAGA: NIE deklarujemy tutaj $configFactory – jest już w ControllerBase.
  protected LandingiApiClient $apiClient;

  public function __construct(LandingiApiClient $api_client, ConfigFactoryInterface $config_factory) {
    $this->apiClient = $api_client;
    // To pole jest zdefiniowane w klasie bazowej ControllerBase.
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('landingi_landing_pages.api_client'),
      $container->get('config.factory')
    );
  }

/**
 * Lista landingów z API Landingi (z paginacją).
 */
public function listLandings(Request $request): array {
  $page = max(1, (int) $request->query->get('page', 1));
  $search = trim((string) $request->query->get('q', ''));

  $data = $this->apiClient->fetchLandings($page, $search ?: NULL);

  if ($data === NULL) {
    return [
      '#markup' => $this->t('No data from the Landingi API. Check that the token is valid and that the API is working.'),
    ];
  }

  $landings = $data['landings'] ?? [];

  // Pobierz listę już zaimportowanych landingów z configu.
  $imported_config = $this->configFactory->get('landingi_landing_pages.imported');
  $imported = $imported_config->get('landings') ?? [];

  // Zbuduj mapę ID -> true dla szybkiego sprawdzania.
  $imported_ids = [];
  foreach ($imported as $item) {
    if (!empty($item['id'])) {
      $imported_ids[(string) $item['id']] = TRUE;
    }
  }

  // WIEMY, że API zwraca 10 landingów na stronę (oprócz ostatniej).
  // Lepiej ustawić to na sztywno, niż liczyć z count($landings) na bieżącej stronie.
  $per_page = 10;

  // Łączna liczba landingów z API: "count" => "479"
  $total_count = NULL;
  if (isset($data['count']) && is_numeric($data['count'])) {
    $total_count = (int) $data['count'];
  } elseif (isset($data['landings_count']) && is_numeric($data['landings_count'])) {
    $total_count = (int) $data['landings_count'];
  }

  $total_pages = NULL;
  if ($total_count !== NULL && $per_page > 0) {
    $total_pages = (int) ceil($total_count / $per_page);
  }

  // Na wszelki wypadek: nie pozwól wyjść poza zakres.
  if ($total_pages !== NULL && $page > $total_pages) {
    $page = $total_pages;
  }
  

  $header = [
    $this->t('ID'),
    $this->t('Name'),
    $this->t('Slug'),
    $this->t('Actions'),
  ];

  $rows = [];

  foreach ($landings as $landing) {
    $landing_id = (string) ($landing['id'] ?? '');
    $slug = $landing['slug'] ?? '';

    // Zbuduj query dla linku Import (z zachowaniem page / q).
    $query_import = ['page' => $page];
    if ($search !== '') {
      $query_import['q'] = $search;
    }

    $is_imported = $landing_id !== '' && isset($imported_ids[$landing_id]);

    if ($is_imported) {
      // Już zaimportowany – zamiast linku Import pokazujemy status.
      $import_cell = [
        'data' => [
          '#markup' => '<span class="landingi-import-status">' . $this->t('Imported') . '</span>',
        ],
      ];
    }
    else {
      // Niezaimportowany – normalny przycisk Import.
      $import_url = Url::fromRoute('landingi_landing_pages.admin_import', [
        'landing_id' => $landing_id,
      ], [
        'query' => $query_import,
      ]);

      $import_cell = [
        'data' => [
          '#type' => 'link',
          '#title' => $this->t('Import'),
          '#url' => $import_url,
          '#attributes' => [
            'class' => ['button', 'button--small'],
          ],
        ],
      ];
    }

    $rows[] = [
      $landing_id,
      $landing['name'] ?? '',
      $slug,
      $import_cell,
    ];
  }


  // Numeryczna paginacja: « pierwsza, ‹ poprzednia, 1 2 3 ..., następna ›, ostatnia ».
  $pager_markup = '';

  if ($total_pages !== NULL && $total_pages > 1) {
    $common_query = [];
    if ($search !== '') {
      $common_query['q'] = $search;
    }

    $items = [];

    // Pierwsza «
    if ($page > 1) {
      $query_first = $common_query + ['page' => 1];
      $url_first = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_first])->toString();
      $items[] = '<li class="pager__item pager__item--first"><a href="' . $url_first . '" class="pager__link"></a></li>';
    }

    // Poprzednia ‹
    if ($page > 1) {
      $query_prev = $common_query + ['page' => $page - 1];
      $url_prev = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_prev])->toString();
      $items[] = '<li class="pager__item pager__item--previous"><a href="' . $url_prev . '" class="pager__link"></a></li>';
    }

    // Zakres numerów stron.
    $start = max(1, $page - 3);
    $end = min($total_pages, $page + 3);

    if ($start > 1) {
      $query_p1 = $common_query + ['page' => 1];
      $url_p1 = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_p1])->toString();
      $items[] = '<li class="pager__item"><a href="' . $url_p1 . '" class="pager__link">1</a></li>';

      if ($start > 2) {
        $items[] = '<li class="pager__item pager__item--ellipsis"><span class="pager__item__ellipsis">…</span></li>';
      }
    }

    for ($i = $start; $i <= $end; $i++) {
      if ($i == $page) {
        $items[] = '<li class="pager__item pager__item--current"><span class="pager__link pager__link--is-active">' . $i . '</span></li>';
      }
      else {
        $query_pi = $common_query + ['page' => $i];
        $url_pi = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_pi])->toString();
        $items[] = '<li class="pager__item"><a href="' . $url_pi . '" class="pager__link">' . $i . '</a></li>';
      }
    }

    if ($end < $total_pages) {
      if ($end < $total_pages - 1) {
        $items[] = '<li class="pager__item pager__item--ellipsis"><span class="pager__item__ellipsis">…</span></li>';
      }

      $query_lastnum = $common_query + ['page' => $total_pages];
      $url_lastnum = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_lastnum])->toString();
      $items[] = '<li class="pager__item"><a href="' . $url_lastnum . '" class="pager__link">' . $total_pages . '</a></li>';
    }

    // Następna ›
    if ($page < $total_pages) {
      $query_next = $common_query + ['page' => $page + 1];
      $url_next = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_next])->toString();
      $items[] = '<li class="pager__item pager__item--next"><a href="' . $url_next . '" class="pager__link"></a></li>';
    }

    // Ostatnia »
    if ($page < $total_pages) {
      $query_last = $common_query + ['page' => $total_pages];
      $url_last = Url::fromRoute('landingi_landing_pages.admin_list', [], ['query' => $query_last])->toString();
      $items[] = '<li class="pager__item pager__item--last"><a href="' . $url_last . '" class="pager__link"></a></li>';
    }

    // Używamy klas jak w core'owym pagerze Claro.
    $pager_markup =
      '<nav class="pager" role="navigation" aria-label="Pagination">' .
        '<ul class="pager__items js-pager__items">' .
          implode('', $items) .
        '</ul>' .
      '</nav>';
  }

  $pager = [
    '#type' => 'markup',
    '#markup' => $pager_markup,
    '#allowed_tags' => ['nav', 'ul', 'li', 'a', 'span'],
  ];


  return [
    'table' => [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No landing pages to display.'),
    ],
    'pager' => $pager,
  ];
}



  /**
   * Importuje wskazany landing i zapisuje jego dane w konfiguracji.
   */
  public function import(string $landing_id, Request $request) {
    // Odczytaj numer strony z query, np. ?page=2
    $page = max(1, (int) $request->query->get('page', 1));
  
    // Pobierz listę landingów z TEJ strony
    $data = $this->apiClient->fetchLandings($page);
  
    if ($data === NULL) {
      $this->messenger()->addError($this->t('Failed to retrieve the list of landing pages from the API.'));
      return $this->redirect('landingi_landing_pages.admin_list', [], ['query' => ['page' => $page]]);
    }
  
    $found = NULL;
    foreach ($data['landings'] ?? [] as $landing) {
      if ((string) ($landing['id'] ?? '') === (string) $landing_id) {
        $found = $landing;
        break;
      }
    }
  
    if ($found === NULL) {
      $this->messenger()->addError($this->t('The specified landing page was not found.'));
      return $this->redirect('landingi_landing_pages.admin_list', [], ['query' => ['page' => $page]]);
    }

    $config = $this->configFactory->getEditable('landingi_landing_pages.imported');
    $imported = $config->get('landings') ?? [];

    $slug = $found['slug'] ?? ('landing_' . $landing_id);

    $imported[$slug] = [
      'id' => $found['id'] ?? '',
      'name' => $found['name'] ?? '',
      'hash' => $found['hash'] ?? '',
      'slug' => $slug,
    ];

    $config->set('landings', $imported)->save();

    $this->messenger()->addStatus($this->t(
      'The landing page "@name" has been imported. It will be available at /landingi/@slug.',
      [
        '@name' => $found['name'] ?? $slug,
        '@slug' => $slug,
      ]
    ));

    return $this->redirect('landingi_landing_pages.admin_imported_list');

  }

  /**
   * Lista zaimportowanych landingów z konfiguracji.
   */
  public function listImported(): array {
    // Dane zapisane w import() w configu 'landingi_landing_pages.imported'.
    $config = $this->configFactory->get('landingi_landing_pages.imported');
    $imported = $config->get('landings') ?? [];

    $header = [
      $this->t('ID'),
      $this->t('Name'),
      $this->t('Slug'),
      $this->t('URL'),
      $this->t('Actions'),
    ];

    $rows = [];

    foreach ($imported as $slug => $landing) {
      $frontend_url = Url::fromRoute('landingi_landing_pages.render', ['slug' => $slug], ['absolute' => TRUE]);
    
      $delete_url = Url::fromRoute('landingi_landing_pages.admin_imported_delete', [
        'slug' => $slug,
      ]);
    
      $alias_url = Url::fromRoute('landingi_landing_pages.admin_imported_alias', [
        'slug' => $slug,
      ]);
    
      // Standardowy "operations" dropbutton z dwoma linkami.
      $operations = [
        '#type' => 'operations',
        '#links' => [
          'alias' => [
            'title' => $this->t('Set alias'),
            'url' => $alias_url,
          ],
          'delete' => [
            'title' => $this->t('Delete'),
            'url' => $delete_url,
          ],
        ],
      ];
    
      $rows[] = [
        $landing['id'] ?? '',
        $landing['name'] ?? '',
        $slug,
        [
          'data' => [
            '#type' => 'link',
            '#title' => $frontend_url->toString(),
            '#url' => $frontend_url,
            '#attributes' => [
              'target' => '_blank',
              'rel' => 'noopener noreferrer',
            ],
          ],
        ],
        [
          'data' => $operations,
        ],
      ];
    }
    


    return [
      'table' => [
        '#type' => 'table',
        '#header' => $header,
        '#rows' => $rows,
        '#empty' => $this->t('No imported landing pages.'),
      ],
    ];
  }

  /**
   * Usuwa zaimportowany landing z konfiguracji.
   *
   * @param string $slug
   *   Klucz (slug) landingu w configu.
   */
  public function deleteImported(string $slug) {
    $config = $this->configFactory->getEditable('landingi_landing_pages.imported');
    $imported = $config->get('landings') ?? [];

    if (!isset($imported[$slug])) {
      $this->messenger()->addError($this->t('The specified imported landing page was not found.'));
      return $this->redirect('landingi_landing_pages.admin_imported_list');
    }

    unset($imported[$slug]);
    $config->set('landings', $imported)->save();

    $this->messenger()->addStatus($this->t(
      'Landing  "@slug" has been removed from the imported list.',
      ['@slug' => $slug]
    ));

    return $this->redirect('landingi_landing_pages.admin_imported_list');
  }


}
