<?php

namespace Drupal\landingi_landing_pages\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\landingi_landing_pages\Service\LandingiApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;


class LandingiRenderController extends ControllerBase {

  protected LandingiApiClient $apiClient;
  protected RequestStack $requestStack;
  private const EXPORT_URL = 'https://www.landingiexport.com';

  public function __construct(
    LandingiApiClient $api_client,
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory
  ) {
    $this->apiClient = $api_client;
    $this->requestStack = $request_stack;
    // używamy właściwości z ControllerBase
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('landingi_landing_pages.api_client'),
      $container->get('request_stack'),
      $container->get('config.factory')
    );
  }

  /**
   * Renderuje landing na podstawie sluga zapisanego w konfiguracji.
   */
  public function renderLanding(string $slug): Response {
    // Dane tylko z configu – bez nodów.
    $config = $this->configFactory->get('landingi_landing_pages.imported');
    $imported = $config->get('landings') ?? [];

    if (!isset($imported[$slug])) {
      throw new NotFoundHttpException($this->t('Landing nie został zaimportowany.'));
    }

    $landing = $imported[$slug];

    $request = $this->requestStack->getCurrentRequest();
    $host = $request->getHost();
    $path = $request->getPathInfo();
    $conversion_hash = $request->query->get('hash');

    $result = $this->apiClient->fetchRenderedLanding($landing, $host, $path, $conversion_hash);

    // Redirect z API (301/302).
    if (!empty($result['redirect']) && in_array($result['status_code'], [301, 302], TRUE)) {
      return new Response('', $result['status_code'], [
        'Location' => $result['redirect'],
      ]);
    }

    if ($result['status_code'] !== 200 || empty($result['content'])) {
      return new Response($this->t('Błąd podczas renderowania landing page.'), 500);
    }

    // HTML z API.
    $html = $result['content'];

    // Test ID z API (jeśli jest).
    $tid = $result['tid'] ?? '';

    // URL tej strony (redirect dla formularza / lightboxa).
    $request = $this->requestStack->getCurrentRequest();
    $redirectUrl = $request->getSchemeAndHttpHost() . $request->getPathInfo();

    // Kolejność dokładnie jak w pluginie WP:
    $html = $this->modifyFormAndRedirectInputEndpoints($html, $landing, $redirectUrl, $tid);
    $html = $this->modifyButtonSubmissionEndpoints($html, $landing, $tid);
    $html = $this->injectLightboxJsHandler($html, $landing, $redirectUrl, $tid);
    $html = $this->fixBrokenHtmlTags($html);
    $html = $this->injectAdditionalCss($html);

    return new Response($html, 200, [
      'Content-Type' => 'text/html; charset=utf-8',
    ]);

  }

    /**
   * Podmiana endpointów formularzy i hidden inputa _redirect.
   */
  private function modifyFormAndRedirectInputEndpoints(string $html, array $landing, string $redirectUrl, ?string $tid): string {
    $hash = $landing['hash'] ?? '';

    // 1) action="/coś" -> action="EXPORT_URL/coś?export_hash=HASH&tid=TID"
    $html = preg_replace(
      '/ action="\/([\s\S]*?)"/',
      sprintf(
        ' action="%s/${1}?export_hash=%s&tid=%s"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    // 2) <input type="hidden" name="_redirect" value="">
    //    -> value="http(s)://host/aktualna-sciezka"
    $html = preg_replace(
      '/(<input type="hidden" name="_redirect" value)="">/',
      sprintf('$1="%s">', $redirectUrl),
      $html
    );

    return $html;
  }

    /**
   * Podmiana href dla /button/xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
   * (tak jak w pluginie WP).
   */
  private function modifyButtonSubmissionEndpoints(string $html, array $landing, ?string $tid): string {
    $hash = $landing['hash'] ?? '';

    // Najpierw wersja z lightboxem, potem bez.
    $html = preg_replace(
      '/ href="(?:\/[^\/]+)?(\/button\/[A-Za-z0-9]{32})\?lightbox=([a-z0-9]{8}(?:-[a-z0-9]{4}){3}-[a-z0-9]{12})"/',
      sprintf(
        ' href="%s${1}?export_hash=%s&tid=%s&lightbox=${2}"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    $html = preg_replace(
      '/ href="(?:\/[^\/]+)?(\/button\/[A-Za-z0-9]{32})"/',
      sprintf(
        ' href="%s${1}?export_hash=%s&tid=%s"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    return $html;
  }

    /**
   * Wstrzyknięcie JS handlera do Lightboxa.
   * Prosta wersja: jeśli w HTML jest skrypt "lightbox-handler",
   * dokładamy nasz <script> przed </body>.
   */
  private function injectLightboxJsHandler(string $html, array $landing, string $redirectUrl, ?string $tid): string {
    // Jeśli w ogóle nie ma lightbox-handler, nie ruszamy.
    if (strpos($html, 'lightbox-handler') === FALSE) {
      return $html;
    }

    $hash = $landing['hash'] ?? '';

    $script = sprintf(
      "<script>
if (typeof Lightbox !== 'undefined') {
  Lightbox.init({
    exportUrl: '%s',
    hash: '%s',
    tid: '%s',
    redirectUrl: '%s'
  });
  Lightbox.register();
}
</script>",
      self::EXPORT_URL,
      addslashes($hash),
      addslashes($tid ?? ''),
      addslashes($redirectUrl)
    );

    // Wstawiamy nasz script przed </body>. Jeśli nie ma </body>, doklejamy na koniec.
    if (stripos($html, '</body>') !== FALSE) {
      return preg_replace('/<\/body>/i', $script . '</body>', $html, 1);
    }

    return $html . $script;
  }

    /**
   * Fix jak w WP: poprawka na JS z .replace(), który rozwala parser HTML.
   */
  private function fixBrokenHtmlTags(string $html): string {
    return str_replace(
      ".replace(/, '<')",
      ".replace(/\\</g, '<')",
      $html
    );
  }

    /**
   * Wstrzykuje dodatkowy CSS do <head>.
   *
   * .row { height: inherit; }
   */
  private function injectAdditionalCss(string $html): string {
    $style = "<style>.row { height: inherit; }</style>";

    // Standardowy przypadek – mamy </head>, wstrzykujemy przed nim.
    if (stripos($html, '</head>') !== FALSE) {
      return preg_replace('/<\/head>/i', $style . '</head>', $html, 1);
    }

    // Jeśli nie ma <head>, spróbujmy tuż po <body>.
    if (stripos($html, '<body') !== FALSE) {
      return preg_replace('/<body([^>]*)>/i', '<body$1>' . $style, $html, 1);
    }

    // Awaryjnie – doklej na początek.
    return $style . $html;
  }

}
