<?php

namespace Drupal\landingi_landing_pages\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;

class LandingiApiClient {

  protected ClientInterface $httpClient;
  protected ConfigFactoryInterface $configFactory;
  protected LoggerChannelInterface $logger;

  protected string $apiBaseUrl = 'https://api.landingi.com/';
  protected string $renderBaseUrl = 'https://www.landingiexport.com';

  public function __construct(
    ClientInterface $http_client,
    ConfigFactoryInterface $config_factory,
    LoggerChannelInterface $logger
  ) {
    $this->httpClient = $http_client;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  protected function getApiToken(): ?string {
    return $this->configFactory
      ->get('landingi_landing_pages.settings')
      ->get('api_token');
  }

  /**
   * Uproszczone pobieranie listy landingów.
   *
   * @param int $page
   *   Numer strony (od 1).
   * @param string|null $search
   *   Fraza do wyszukiwania po nazwie (opcjonalnie).
   */
  public function fetchLandings(int $page = 1, ?string $search = NULL): ?array {
    $token = $this->getApiToken();
    if (empty($token)) {
      return NULL;
    }

    $query = [
      'page' => $page,
    ];

    // Jeśli podamy frazę – przekaż ją do API (np. jako "phrase").
    if (!empty($search)) {
      $query['phrase'] = $search;
    }

    $options = [
      'headers' => [
        'apiKey' => $token,
      ],
      'query' => $query,
      'verify' => FALSE,
    ];

    try {
      $response = $this->httpClient->request('GET', $this->apiBaseUrl . 'wordpress/landings', $options);
      $body = $response->getBody()->getContents();

      $data = json_decode($body, TRUE);
      if (!is_array($data)) {
        $this->logger->error('Landingi API – nieprawidłowy JSON: @body', ['@body' => $body]);
        return NULL;
      }

      return $data;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Landingi API error (list): @msg', ['@msg' => $e->getMessage()]);
      return NULL;
    }
  }



  /**
   * Pobiera wyrenderowany landing (HTML) z eksportera.
   */
  public function fetchRenderedLanding(array $landing, string $host, string $path, ?string $conversion_hash = NULL): array {
    $query = [
      'export_hash' => $landing['hash'] ?? '',
      'tid' => $landing['testId'] ?? '',
    ];

    if ($conversion_hash) {
      $query['conversion_hash'] = $conversion_hash;
    }

    $headers = [
      'X-export-source' => 'drupal',
      'X-export-host' => $host,
      'X-export-path' => $path,
      'User-Agent' => sprintf('Landingi Drupal Module PHP/%s', PHP_VERSION),
    ];

    try {
      $response = $this->httpClient->request('GET', $this->renderBaseUrl . '/api/render', [
        'headers' => $headers,
        'query' => $query,
        'verify' => FALSE,
      ]);

      $status = $response->getStatusCode();
      $body = $response->getBody()->getContents();

      // Próba JSON-a; jeśli nie, traktujemy jako czysty HTML.
      $data = json_decode($body, TRUE);
      if (json_last_error() === JSON_ERROR_NONE && is_array($data)) {
        return [
          'status_code' => $status,
          'content' => $data['content'] ?? NULL,
          'redirect' => $data['redirect'] ?? NULL,
          'tid' => $data['tid'] ?? NULL,
        ];
      }

      return [
        'status_code' => $status,
        'content' => $body,
        'redirect' => NULL,
        'tid' => NULL,
      ];
    }
    catch (GuzzleException $e) {
      $this->logger->error('Landingi API error (render): @msg', ['@msg' => $e->getMessage()]);
      return [
        'status_code' => 500,
        'content' => NULL,
        'redirect' => NULL,
        'tid' => NULL,
      ];
    }
  }

}
