<?php

namespace Drupal\landingi_landing_pages\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\path_alias\Entity\PathAlias;

/**
 * Simple form used to set an alias for /landingi/{slug}.
 */
class LandingiAliasForm extends FormBase {

  /**
   * @var string
   */
  protected $slug;

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'landingi_landing_pages_alias_form';
  }

  /**
   * {@inheritdoc}
   *
   * @param string $slug
   *   The {slug} route parameter.
   */
  public function buildForm(array $form, FormStateInterface $form_state, string $slug = NULL): array {
    $this->slug = $slug;

    $system_path = '/landingi/' . $slug;
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    // Look for an existing alias for this path.
    $storage = \Drupal::entityTypeManager()->getStorage('path_alias');
    $existing = $storage->loadByProperties([
      'path' => $system_path,
      'langcode' => $langcode,
    ]);
    $alias_entity = $existing ? reset($existing) : NULL;
    $current_alias = $alias_entity ? $alias_entity->getAlias() : '';

    $form['info'] = [
      '#markup' => $this->t(
        'System path: <code>@path</code><br>Set an alias URL for this landing page.',
        ['@path' => $system_path]
      ),
    ];

    $form['alias'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL alias'),
      '#default_value' => $current_alias ?: '/' . $slug,
      '#description' => $this->t('Enter an alias starting with "/". For example, "/my-first-lp". Leave blank to remove the alias.'),
      '#required' => FALSE,
    ];

    // Pass the slug through submit.
    $form['slug'] = [
      '#type' => 'hidden',
      '#value' => $slug,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save alias'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $slug = $form_state->getValue('slug');
    $alias = trim($form_state->getValue('alias') ?? '');

    $system_path = '/landingi/' . $slug;
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $storage = \Drupal::entityTypeManager()->getStorage('path_alias');
    $existing = $storage->loadByProperties([
      'path' => $system_path,
      'langcode' => $langcode,
    ]);
    $alias_entity = $existing ? reset($existing) : NULL;

    if ($alias === '') {
      // Empty field means remove the alias.
      if ($alias_entity) {
        $alias_entity->delete();
      }
      $this->messenger()->addStatus($this->t('The alias for "@path" has been removed.', [
        '@path' => $system_path,
      ]));
    }
    else {
      // Make sure the alias starts with "/".
      if ($alias[0] !== '/') {
        $alias = '/' . $alias;
      }

      if ($alias_entity) {
        $alias_entity->setAlias($alias);
        $alias_entity->save();
      }
      else {
        $alias_entity = PathAlias::create([
          'path' => $system_path,
          'alias' => $alias,
          'langcode' => $langcode,
        ]);
        $alias_entity->save();
      }

      $this->messenger()->addStatus($this->t(
        'The alias "@alias" has been assigned to the path "@path".',
        [
          '@alias' => $alias,
          '@path' => $system_path,
        ]
      ));
    }

    // After saving redirect back to the list of imported landings.
    $form_state->setRedirect('landingi_landing_pages.admin_imported_list');
  }

}
