<?php

namespace Drupal\landingi_landing_pages\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\landingi_landing_pages\Service\LandingiApiClient;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;


class LandingiRenderController extends ControllerBase {

  protected LandingiApiClient $apiClient;
  protected RequestStack $requestStack;
  private const EXPORT_URL = 'https://www.landingiexport.com';

  public function __construct(
    LandingiApiClient $api_client,
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory
  ) {
    $this->apiClient = $api_client;
    $this->requestStack = $request_stack;
    // Use the property that already exists in ControllerBase.
    $this->configFactory = $config_factory;
  }

  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('landingi_landing_pages.api_client'),
      $container->get('request_stack'),
      $container->get('config.factory')
    );
  }

  /**
   * Renders a landing page based on the slug stored in configuration.
   */
  public function renderLanding(string $slug): Response {
    // Data comes only from config; nodes are not used.
    $config = $this->configFactory->get('landingi_landing_pages.imported');
    $imported = $config->get('landings') ?? [];

    if (!isset($imported[$slug])) {
      throw new NotFoundHttpException($this->t('Landing nie został zaimportowany.'));
    }

    $landing = $imported[$slug];

    $request = $this->requestStack->getCurrentRequest();
    $host = $request->getHost();
    $path = $request->getPathInfo();
    $conversion_hash = $request->query->get('hash');

    $result = $this->apiClient->fetchRenderedLanding($landing, $host, $path, $conversion_hash);

    // Redirect returned by the API (301/302).
    if (!empty($result['redirect']) && in_array($result['status_code'], [301, 302], TRUE)) {
      return new Response('', $result['status_code'], [
        'Location' => $result['redirect'],
      ]);
    }

    if ($result['status_code'] !== 200 || empty($result['content'])) {
      return new Response($this->t('Błąd podczas renderowania landing page.'), 500);
    }

    // HTML returned by the API.
    $html = $result['content'];

    // Test ID returned by the API, if available.
    $tid = $result['tid'] ?? '';

    // URL of the current page (used for form/lightbox redirects).
    $request = $this->requestStack->getCurrentRequest();
    $redirectUrl = $request->getSchemeAndHttpHost() . $request->getPathInfo();

    // Apply transformations in the same order as the WP plugin:
    $html = $this->modifyFormAndRedirectInputEndpoints($html, $landing, $redirectUrl, $tid);
    $html = $this->modifyButtonSubmissionEndpoints($html, $landing, $tid);
    $html = $this->injectLightboxJsHandler($html, $landing, $redirectUrl, $tid);
    $html = $this->fixBrokenHtmlTags($html);
    $html = $this->injectAdditionalCss($html);

    return new Response($html, 200, [
      'Content-Type' => 'text/html; charset=utf-8',
    ]);

  }

  /**
   * Rewrites form endpoints and the hidden _redirect input.
   */
  private function modifyFormAndRedirectInputEndpoints(string $html, array $landing, string $redirectUrl, ?string $tid): string {
    $hash = $landing['hash'] ?? '';

    // 1) action="/..." -> action="EXPORT_URL/.../?export_hash=HASH&tid=TID"
    $html = preg_replace(
      '/ action="\/([\s\S]*?)"/',
      sprintf(
        ' action="%s/${1}?export_hash=%s&tid=%s"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    // 2) <input type="hidden" name="_redirect" value="">
    //    -> value="http(s)://host/current-path"
    $html = preg_replace(
      '/(<input type="hidden" name="_redirect" value)="">/',
      sprintf('$1="%s">', $redirectUrl),
      $html
    );

    return $html;
  }

  /**
   * Rewrites href attributes for /button/xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx
   * (same approach as the WP plugin).
   */
  private function modifyButtonSubmissionEndpoints(string $html, array $landing, ?string $tid): string {
    $hash = $landing['hash'] ?? '';

    // Handle the lightbox variant first, then the plain version.
    $html = preg_replace(
      '/ href="(?:\/[^\/]+)?(\/button\/[A-Za-z0-9]{32})\?lightbox=([a-z0-9]{8}(?:-[a-z0-9]{4}){3}-[a-z0-9]{12})"/',
      sprintf(
        ' href="%s${1}?export_hash=%s&tid=%s&lightbox=${2}"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    $html = preg_replace(
      '/ href="(?:\/[^\/]+)?(\/button\/[A-Za-z0-9]{32})"/',
      sprintf(
        ' href="%s${1}?export_hash=%s&tid=%s"',
        self::EXPORT_URL,
        $hash,
        $tid ?? ''
      ),
      $html
    );

    return $html;
  }

  /**
   * Injects the JS handler for the lightbox.
   * Simple version: if the HTML contains the "lightbox-handler" script,
   * append our <script> right before </body>.
   */
  private function injectLightboxJsHandler(string $html, array $landing, string $redirectUrl, ?string $tid): string {
    // Skip if lightbox-handler is not present at all.
    if (strpos($html, 'lightbox-handler') === FALSE) {
      return $html;
    }

    $hash = $landing['hash'] ?? '';

    $script = sprintf(
      "<script>
if (typeof Lightbox !== 'undefined') {
  Lightbox.init({
    exportUrl: '%s',
    hash: '%s',
    tid: '%s',
    redirectUrl: '%s'
  });
  Lightbox.register();
}
</script>",
      self::EXPORT_URL,
      addslashes($hash),
      addslashes($tid ?? ''),
      addslashes($redirectUrl)
    );

    // Insert the script before </body>; append to the end if </body> is missing.
    if (stripos($html, '</body>') !== FALSE) {
      return preg_replace('/<\/body>/i', $script . '</body>', $html, 1);
    }

    return $html . $script;
  }

  /**
   * Same fix as WP: offsets JS .replace() that breaks the HTML parser.
   */
  private function fixBrokenHtmlTags(string $html): string {
    return str_replace(
      ".replace(/, '<')",
      ".replace(/\\</g, '<')",
      $html
    );
  }

  /**
   * Injects additional CSS into <head>.
   *
   * .row { height: inherit; }
   */
  private function injectAdditionalCss(string $html): string {
    $style = "<style>.row { height: inherit; }</style>";

    // Happy path: </head> is present, so inject right before it.
    if (stripos($html, '</head>') !== FALSE) {
      return preg_replace('/<\/head>/i', $style . '</head>', $html, 1);
    }

    // If <head> is missing, try inserting right after <body>.
    if (stripos($html, '<body') !== FALSE) {
      return preg_replace('/<body([^>]*)>/i', '<body$1>' . $style, $html, 1);
    }

    // Fallback: prepend the style block at the beginning.
    return $style . $html;
  }

}
