<?php

declare(strict_types=1);

namespace Drupal\Tests\langfuse\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\langfuse\LangFuseClientInterface;

/**
 * Tests LangFuse configuration and service initialization.
 *
 * @group langfuse
 */
class LangFuseConfigTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'config',
    'langfuse',
  ];

  /**
   * The LangFuse client service.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langFuseClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install relevant schemas.
    $this->installEntitySchema('user');
    $this->installSchema('system', ['sequences']);
    $this->installConfig('langfuse');

    // Retrieve the client service.
    $this->langFuseClient = $this->container->get('langfuse.client');

    // Ensure we have a default auth_method so the client won't fail on load.
    if (!$this->config('langfuse.settings')->get('auth_method')) {
      $this->config('langfuse.settings')
        ->set('auth_method', 'none')
        ->save();
    }
  }

  /**
   * Tests default configuration matches what is in config/install.
   */
  public function testDefaultConfig(): void {
    $config = $this->config('langfuse.settings');

    // Check defaults from config/install/langfuse.settings.yml.
    $this->assertNull($config->get('langfuse_url'));
    $this->assertEquals('none', $config->get('auth_method'));
    $this->assertNull($config->get('bearer_token'));

    $basicAuth = $config->get('basic_auth');
    $this->assertIsArray($basicAuth);
    $this->assertArrayHasKey('username', $basicAuth);
    $this->assertArrayHasKey('password', $basicAuth);

    $keyPair = $config->get('key_pair');
    $this->assertIsArray($keyPair);
    $this->assertArrayHasKey('public_key', $keyPair);
    $this->assertArrayHasKey('secret_key', $keyPair);

    // Confirm that the client is not configured by default.
    $this->assertFalse($this->langFuseClient->isConfigured(), 'Client is not configured with defaults.');
  }

  /**
   * Tests that updating the config dynamically affects the client service.
   */
  public function testClientConfig(): void {
    // Initially unconfigured.
    $this->assertFalse($this->langFuseClient->isConfigured());

    // Configure the client with a key pair.
    $this->config('langfuse.settings')
      ->set('langfuse_url', 'https://cloud.langfuse.com')
      ->set('auth_method', 'key_pair')
      ->set('key_pair', [
        'public_key' => 'test-public-key',
        'secret_key' => 'test-secret-key',
      ])
      ->save();

    // Now the client should be configured.
    $this->assertTrue($this->langFuseClient->isConfigured(), 'Client is configured with new config values.');

    // Verify the config array matches the newly set values.
    $config = $this->langFuseClient->getConfig();
    $this->assertEquals('https://cloud.langfuse.com', $config['langfuse_url']);
    $this->assertEquals('key_pair', $config['auth_method']);
    $this->assertEquals('test-public-key', $config['key_pair']['public_key']);
    $this->assertEquals('test-secret-key', $config['key_pair']['secret_key']);
  }

}
