# LangFuse Example Module

This submodule provides an example implementation demonstrating how to integrate 
LangFuse observability with OpenAI in Drupal. It serves as a reference for 
developers wanting to implement LangFuse tracking in their own AI-powered 
Drupal applications.

> **Note:** This is an example submodule bundled with the langfuse module.
> It is not meant to be installed separately with Composer.

## Overview

This example demonstrates a simple AI integration with these features:
- Configuration form for OpenAI settings
- Interactive chat form with LangFuse tracing
- Complete logging of AI interactions with spans, generations, and statistics
- Token usage tracking

Unlike the core Drupal AI module approach, this example uses the OpenAI PHP 
client directly to demonstrate a custom integration scenario.

## Setup Instructions

### 1. Configure the LangFuse Module

Before using this example:
1. Install and enable the parent LangFuse module
2. Navigate to `/admin/config/system/langfuse/settings`
3. Enter your LangFuse API credentials (public & secret keys)
4. Set other desired LangFuse configuration options
5. Test the connection to ensure LangFuse is properly set up

### 2. Configure OpenAI Settings

1. Navigate to `/admin/config/system/langfuse/openai-settings` 
2. Enter your OpenAI API key
3. Optionally enter your organization ID
4. Select default model (GPT-4, GPT-4o-mini, or GPT-3.5 Turbo)
5. Test the connection to ensure OpenAI access works

### 3. Use the Demo Form

1. Visit `/admin/config/system/langfuse/openai-demo`
2. Enter a prompt in the text area
3. Adjust model and temperature settings if desired
4. Submit the form to generate a response
5. View the response along with:
   - LangFuse trace ID
   - Token usage statistics
   - Request timing information

### 4. Examine Traces in LangFuse Dashboard

1. Log into your LangFuse dashboard
2. Find the traces generated by the example module
3. Analyze detailed information about:
   - Prompts and responses
   - Token usage
   - Timing
   - Quality scores

## Developer Integration Guide

### Basic Lifecycle of LangFuse Tracing

#### Before LangFuse Integration

A simple OpenAI API call might look like:

```php
// Create the OpenAI client
$client = OpenAI::factory()
  ->withApiKey($api_key)
  ->make();

// Make API request
$response = $client->chat()->create([
  'model' => 'gpt-4o-mini',
  'messages' => [['role' => 'user', 'content' => $prompt]],
]);

// Process response
$content = $response->choices[0]->message->content;
```

#### After LangFuse Integration

With LangFuse tracing, the workflow becomes:

```php
// Get LangFuse client
$langfuse_client = \Drupal::service('langfuse.client');

// 1. Create a trace to represent the entire operation
$trace = $langfuse_client->createTrace(
  'my_ai_operation',
  \Drupal::currentUser()->id(),
  session_id(),
  ['context' => 'additional data']
);

// 2. Create a span for the specific API call
$span = $trace->createSpan(
  'openai_request',
  ['model' => 'gpt-4o-mini']
);

// 3. Track the generation specifically
$generation = $span->createGeneration(
  'chat_completion',
  'gpt-4o-mini',
  ['temperature' => 0.7],
  ['purpose' => 'example'],
  [['role' => 'user', 'content' => $prompt]]
);

// Make API request - same as before
$response = $client->chat()->create([
  'model' => 'gpt-4o-mini',
  'messages' => [['role' => 'user', 'content' => $prompt]],
]);
$content = $response->choices[0]->message->content;

// 4. Complete the generation tracking
$generation->end([
  'output' => $content,
  'usage_details' => [
    'prompt_tokens' => $response->usage->prompt_tokens,
    'completion_tokens' => $response->usage->completion_tokens,
    'total_tokens' => $response->usage->total_tokens,
  ],
]);

// 5. End the span - timing is tracked automatically
$span->end(['status' => 'success']);

// 6. Optionally add scores
$trace->score('helpfulness', 0.95);

// 7. Traces will sync automatically during kernel termination
// To force immediate sync:
// $langfuse_client->syncTraces(TRUE);
```

### Automatic Trace Synchronization

By default, traces are synchronized to LangFuse during Drupal's kernel 
termination phase. This ensures that all data is sent to LangFuse after the 
page response has been sent to the user, preventing any impact on page 
performance.

For non-web contexts (like Drush commands), you may need to explicitly call 
`syncTraces()`:

```php
// At the end of a CLI operation
$langfuse_client->syncTraces(TRUE);
```

## Advanced Usage

### Coordinating Traces Across Multiple Services

One of the key benefits of LangFuse is the ability to maintain a single coherent 
trace across different services or components of your application:

```php
// Service 1 - Initial service starting the interaction
$langfuse_client = \Drupal::service('langfuse.client');
$trace = $langfuse_client->createTrace('multi_service_workflow');

// The trace becomes the "current active trace" automatically
// No need to explicitly store IDs - the client maintains this state

// Service 2 - Another component of your application
$langfuse_client = \Drupal::service('langfuse.client');
// Get the active trace that was set by Service 1
if ($trace = $langfuse_client->getCurrentTrace()) {
  // Add to the same trace created by Service 1
  $span = $trace->createSpan('service_2_operation');
  // Perform operations...
  $span->end();
}

// Service 3 - Yet another component
$langfuse_client = \Drupal::service('langfuse.client');
// Also gets the same active trace
if ($trace = $langfuse_client->getCurrentTrace()) {
  $trace->createEvent('service_3_event', ['details' => 'Some information']);
}

// When the workflow is complete
$langfuse_client->clearCurrentTrace();
```

### Working with the Current Active Trace

The LangFuse client maintains a concept of the "current" trace for convenience:

```php
// Get the current trace if it exists, or create a new one
$trace = $langfuse_client->getCurrentTrace() ?? 
  $langfuse_client->createTrace('new_operation');

// Add to the trace
$trace->createEvent('user_action', ['action' => 'clicked_button']);

// Clear the current trace when done with a workflow
$langfuse_client->clearCurrentTrace();
```

### Adding Custom Scores and Feedback

You can add custom scores to measure quality or gather user feedback:

```php
// Add scores (value between 0-1)
$trace->score('accuracy', 0.85, 'AI response factual accuracy');
$trace->score('relevance', 0.92, 'Response relevance to query');

// Add user feedback (e.g., from a feedback form)
if ($user_rating) {
  $trace->score(
    'user_satisfaction', 
    $user_rating / 5, 
    'User provided rating'
  );
}
```

## Troubleshooting

* If traces aren't appearing in LangFuse, check your API credentials
* Make sure `syncTraces()` is being called (happens automatically for web 
requests)
* For CLI scripts, explicitly call `syncTraces(TRUE)` at the end
* Check Drupal logs for any LangFuse API errors
* Verify OpenAI API key permissions if getting authorization errors

## See Also

* [LangFuse Documentation](https://langfuse.com/docs)
* [OpenAI PHP Client](https://github.com/openai-php/client)
* [Drupal AI Module](https://www.drupal.org/project/ai)

## License

This project is licensed under the GNU General Public License v2.0 or later.
