<?php

namespace Drupal\langfuse_example\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;

/**
 * Configure OpenAI settings for LangFuse example.
 */
class OpenAiSettingsForm extends ConfigFormBase {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $httpClient;

  /**
   * Constructs a new OpenAiSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \GuzzleHttp\Client $http_client
   *   The HTTP client.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    Client $http_client,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('http_client'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['langfuse_example.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'langfuse_example_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('langfuse_example.settings');

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAI API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('Your OpenAI API key (starts with sk-). Get it from <a href="https://platform.openai.com/api-keys" target="_blank">your OpenAI account</a>.'),
      '#required' => TRUE,
      '#maxlength' => 200,
    ];

    $form['organization'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAI Organization ID'),
      '#default_value' => $config->get('organization'),
      '#description' => $this->t('Optional: Your OpenAI organization ID (starts with org-).'),
      '#maxlength' => 200,
    ];

    $form['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Model'),
      '#options' => [
        'gpt-4o-mini' => $this->t('GPT-4o Mini'),
        'gpt-4.1-mini' => $this->t('GPT-4.1 Mini'),
      ],
      '#default_value' => $config->get('model') ?: 'gpt-4o-mini',
      '#description' => $this->t('Select the default OpenAI model to use.'),
      '#required' => TRUE,
    ];

    $form['test_connection'] = [
      '#type' => 'button',
      '#value' => $this->t('Test Connection'),
      '#ajax' => [
        'callback' => '::testConnectionAjax',
        'wrapper' => 'connection-result',
        'progress' => [
          'type' => 'throbber',
          'message' => $this->t('Testing connection...'),
        ],
      ],
      '#limit_validation_errors' => [],
      '#attributes' => [
        'class' => ['button', 'button--secondary'],
      ],
    ];

    $form['connection_result'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'connection-result'],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback for testing the connection.
   */
  public function testConnectionAjax(array &$form, FormStateInterface $form_state) {
    $api_key = $form_state->getValue('api_key');
    $organization = $form_state->getValue('organization');

    // Create the response container.
    $response = [
      '#type' => 'container',
      '#attributes' => ['id' => 'connection-result'],
    ];

    // Validate API key format before testing.
    if (empty($api_key)) {
      $response['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('Please enter an API key before testing.') . '</div>',
      ];
      return $response;
    }

    if (!str_starts_with($api_key, 'sk-')) {
      $response['message'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--error">' . $this->t('API key should start with "sk-".') . '</div>',
      ];
      return $response;
    }

    // Test the connection.
    $result = $this->testOpenAiConnection($api_key, $organization);

    $response['message'] = [
      '#type' => 'markup',
      '#markup' => $result['success']
        ? '<div class="messages messages--status">' . $this->t('✅ Connection successful! API key is valid.') . '</div>'
        : '<div class="messages messages--error">' . $this->t('❌ Connection failed: @error', ['@error' => $result['message']]) . '</div>',
    ];

    return $response;
  }

  /**
   * Tests the connection to OpenAI API.
   *
   * @param string $api_key
   *   The OpenAI API key.
   * @param string|null $organization
   *   The OpenAI organization ID.
   *
   * @return array
   *   An array with keys 'success' (boolean) and 'message' (string).
   */
  protected function testOpenAiConnection($api_key, $organization = NULL) {
    try {
      $headers = [
        'Authorization' => 'Bearer ' . $api_key,
        'Content-Type' => 'application/json',
        'User-Agent' => 'Drupal-LangFuse-Module/1.0',
      ];

      if (!empty($organization)) {
        $headers['OpenAI-Organization'] = $organization;
      }

      // Use a simpler endpoint that's more reliable for testing.
      // Don't throw exceptions for HTTP errors to handle them gracefully.
      $response = $this->httpClient->request('GET', 'https://api.openai.com/v1/models', [
        'headers' => $headers,
        'timeout' => 15,
        'connect_timeout' => 10,
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();

      if ($status_code === 200) {
        // Parse response to validate it's actually working.
        $data = json_decode($response_body, TRUE);
        if (isset($data['data']) && is_array($data['data']) && count($data['data']) > 0) {
          return [
            'success' => TRUE,
            'message' => 'Connection successful - Found ' . count($data['data']) . ' available models.',
          ];
        }
        else {
          return [
            'success' => FALSE,
            'message' => 'API responded but returned unexpected data format.',
          ];
        }
      }
      elseif ($status_code === 401) {
        return [
          'success' => FALSE,
          'message' => 'Invalid API key. Please check your OpenAI API key.',
        ];
      }
      elseif ($status_code === 403) {
        return [
          'success' => FALSE,
          'message' => 'Access forbidden. Check your API key permissions and organization ID.',
        ];
      }
      elseif ($status_code === 429) {
        return [
          'success' => FALSE,
          'message' => 'Rate limit exceeded. Please try again later.',
        ];
      }
      else {
        return [
          'success' => FALSE,
          'message' => 'HTTP ' . $status_code . ': ' . substr($response_body, 0, 200),
        ];
      }
    }
    catch (GuzzleException $e) {
      // Handle network/connection errors.
      $error_message = $e->getMessage();

      if (str_contains($error_message, 'timeout')) {
        return [
          'success' => FALSE,
          'message' => 'Connection timeout. Please check your internet connection.',
        ];
      }
      elseif (str_contains($error_message, 'Could not resolve host')) {
        return [
          'success' => FALSE,
          'message' => 'DNS resolution failed. Please check your internet connection.',
        ];
      }
      else {
        return [
          'success' => FALSE,
          'message' => 'Network error: ' . $error_message,
        ];
      }
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => 'Unexpected error: ' . $e->getMessage(),
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    $api_key = $form_state->getValue('api_key');
    $organization = $form_state->getValue('organization');

    // Validate API key format.
    if (!empty($api_key) && !str_starts_with($api_key, 'sk-')) {
      $form_state->setErrorByName('api_key', $this->t('OpenAI API key must start with "sk-".'));
    }

    // Validate organization ID format if provided.
    if (!empty($organization) && !str_starts_with($organization, 'org-')) {
      $form_state->setErrorByName('organization', $this->t('OpenAI Organization ID must start with "org-".'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('langfuse_example.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('organization', $form_state->getValue('organization'))
      ->set('model', $form_state->getValue('model'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
