<?php

namespace Drupal\langfuse;

use Dropsolid\LangFuse\Client as LangFuseSdkClient;
use Dropsolid\LangFuse\Observability\Trace;

/**
 * Interface for the LangFuse client service.
 */
interface LangFuseClientInterface {

  /**
   * Gets the SDK client instance.
   *
   * @return \Dropsolid\LangFuse\Client
   *   The SDK client instance.
   *
   * @throws \RuntimeException
   *   When the client cannot be initialized.
   */
  public function getSdkClient(): LangFuseSdkClient;

  /**
   * Gets the configurations.
   *
   * @return array
   *   The configuration array..
   */
  public function getConfig(): array;

  /**
   * Checks if the client is properly configured.
   *
   * @return bool
   *   TRUE if configured, FALSE otherwise.
   */
  public function isConfigured(): bool;

  /**
   * Creates a new trace.
   *
   * @param string $name
   *   The trace name.
   * @param string|null $userId
   *   Optional user ID.
   * @param string|null $sessionId
   *   Optional session ID.
   * @param array|null $metadata
   *   Optional metadata.
   * @param array $tags
   *   Optional tags.
   * @param string|null $version
   *   Optional version.
   * @param string|null $release
   *   Optional release.
   *
   * @return \Dropsolid\LangFuse\Observability\Trace
   *   The created trace.
   */
  public function createTrace(
    string $name,
    ?string $userId = NULL,
    ?string $sessionId = NULL,
    ?array $metadata = NULL,
    array $tags = [],
    ?string $version = NULL,
    ?string $release = NULL,
  ): Trace;

  /**
   * Gets a trace by ID.
   *
   * @param string $traceId
   *   The trace ID.
   *
   * @return \Dropsolid\LangFuse\Observability\Trace|null
   *   The trace, or NULL if not found.
   */
  public function getTrace(string $traceId): ?Trace;

  /**
   * Gets the current active trace.
   *
   * @return \Dropsolid\LangFuse\Observability\Trace|null
   *   The current trace, or NULL if none is active.
   */
  public function getCurrentTrace(): ?Trace;

  /**
   * Sets the current active trace.
   *
   * @param string $traceId
   *   The trace ID to set as current.
   */
  public function setCurrentTrace(string $traceId): void;

  /**
   * Clears the current active trace.
   */
  public function clearCurrentTrace(): void;

  /**
   * Synchronizes pending traces with LangFuse.
   *
   * @param bool $force
   *   Whether to force sync regardless of conditions.
   */
  public function syncTraces(bool $force = FALSE): void;

  /**
   * Tests the connection with given configuration.
   *
   * @param array $config
   *   Configuration array containing connection details.
   *
   * @return bool
   *   TRUE if connection is successful, FALSE otherwise.
   */
  public function checkConnection(array $config = []): bool;

}
