<?php

declare(strict_types=1);

namespace Drupal\langfuse\Form;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\langfuse\LangFuseClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a configuration form for LangFuse settings.
 */
class LangFuseSettingsForm extends ConfigFormBase {

  /**
   * The LangFuse client service.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langFuseClient;

  /**
   * Constructs a new LangFuseSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typedConfigManager
   *   The typed config manager.
   * @param \Drupal\langfuse\LangFuseClientInterface $langFuseClient
   *   The LangFuse client service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    LangFuseClientInterface $langFuseClient,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
    $this->langFuseClient = $langFuseClient;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): LangFuseSettingsForm {
    return new self(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('langfuse.client')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['langfuse.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'langfuse_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('langfuse.settings');

    $form['langfuse_url'] = [
      '#type' => 'url',
      '#title' => $this->t('LangFuse URL'),
      '#description' => $this->t('The URL of your LangFuse instance (e.g., https://cloud.langfuse.com)'),
      '#default_value' => $config->get('langfuse_url'),
      '#required' => TRUE,
    ];

    $form['auth_method'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Method'),
      '#description' => $this->t('Select the authentication method to use with LangFuse API.'),
      '#options' => [
        'none' => $this->t('No Authentication'),
        'bearer' => $this->t('Bearer Token'),
        'basic' => $this->t('Basic Auth'),
        'key_pair' => $this->t('Public/Secret Key'),
      ],
      '#default_value' => $config->get('auth_method') ?? 'none',
      '#required' => TRUE,
    ];

    $this->buildBearerTokenFields($form, $config);
    $this->buildBasicAuthFields($form, $config);
    $this->buildKeyPairFields($form, $config);

    $form['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout'),
      '#description' => $this->t('Specify the request timeout in seconds (0.1 to 300). Default is 10.'),
      '#default_value' => $config->get('timeout') ?? 10.0,
      '#min' => 0.1,
      '#max' => 300,
      '#step' => 0.1,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Builds the bearer token form fields.
   *
   * @param array &$form
   *   The form array to enhance.
   * @param \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig $config
   *   The configuration object containing default values.
   */
  protected function buildBearerTokenFields(array &$form, Config|ImmutableConfig $config): void {
    $form['bearer_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Bearer Token'),
      '#description' => $this->t('Enter your Bearer token (e.g., Bearer xxxxx...)'),
      '#default_value' => $config->get('bearer_token'),
      '#states' => [
        'visible' => [
          ':input[name="auth_method"]' => ['value' => 'bearer'],
        ],
        'required' => [
          ':input[name="auth_method"]' => ['value' => 'bearer'],
        ],
      ],
    ];
  }

  /**
   * Builds the basic authentication form fields.
   *
   * @param array &$form
   *   The form array to enhance.
   * @param \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig $config
   *   The configuration object containing default values.
   */
  protected function buildBasicAuthFields(array &$form, Config|ImmutableConfig $config): void {
    $form['basic_auth'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Basic Authentication'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="auth_method"]' => ['value' => 'basic'],
        ],
      ],
    ];

    $form['basic_auth']['username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Username'),
      '#default_value' => $config->get('basic_auth.username'),
      '#states' => [
        'required' => [
          ':input[name="auth_method"]' => ['value' => 'basic'],
        ],
      ],
    ];

    $form['basic_auth']['password'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Password'),
      '#states' => [
        'required' => [
          ':input[name="auth_method"]' => ['value' => 'basic'],
        ],
      ],
    ];
  }

  /**
   * Builds the API key pair form fields.
   *
   * @param array &$form
   *   The form array to enhance.
   * @param \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig $config
   *   The configuration object containing default values.
   */
  protected function buildKeyPairFields(array &$form, Config|ImmutableConfig $config): void {
    $form['key_pair'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('API Keys'),
      '#description' => $this->t('Enter your LangFuse API key pair.'),
      '#tree' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="auth_method"]' => ['value' => 'key_pair'],
        ],
      ],
    ];

    $form['key_pair']['public_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public Key'),
      '#default_value' => $config->get('key_pair.public_key'),
      '#states' => [
        'required' => [
          ':input[name="auth_method"]' => ['value' => 'key_pair'],
        ],
      ],
    ];

    $form['key_pair']['secret_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret Key'),
      '#default_value' => $config->get('key_pair.secret_key'),
      '#states' => [
        'required' => [
          ':input[name="auth_method"]' => ['value' => 'key_pair'],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate LangFuse URL format.
    $langfuse_url = $form_state->getValue('langfuse_url');
    if (!empty($langfuse_url) && !filter_var($langfuse_url, FILTER_VALIDATE_URL)) {
      $form_state->setError($form['langfuse_url'], $this->t('Please enter a valid URL.'));
    }

    // Validate timeout value.
    $timeout = $form_state->getValue('timeout');
    if (!is_numeric($timeout) || $timeout < 0.1 || $timeout > 300) {
      $form_state->setError($form['timeout'], $this->t('Timeout must be between 0.1 and 300 seconds.'));
    }

    $auth_method = $form_state->getValue('auth_method');
    if ($auth_method === 'none') {
      return;
    }

    try {
      $this->validateAuthenticationMethod($form, $form_state);
    }
    catch (\Exception $e) {
      $form_state->setError(
        $form[$auth_method === 'key_pair' ? 'key_pair' : 'langfuse_url'],
        $this->t('Connection test failed: @error', ['@error' => $e->getMessage()])
      );
    }
  }

  /**
   * Validates the selected authentication method and its credentials.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @throws \Exception
   *   Throws exception if validation fails.
   */
  protected function validateAuthenticationMethod(array &$form, FormStateInterface $form_state): void {
    // Skip validation if configured to do so.
    if ($this->config('langfuse.settings')->get('skip_validation')) {
      return;
    }

    $auth_method = $form_state->getValue('auth_method');
    $config = ['host' => $form_state->getValue('langfuse_url')];

    switch ($auth_method) {
      case 'key_pair':
        $key_pair = $form_state->getValue('key_pair');
        if (empty($key_pair['public_key']) || empty($key_pair['secret_key'])) {
          $form_state->setError($form['key_pair'], $this->t('Both Public and Secret keys are required.'));
          return;
        }
        $config['public_key'] = $key_pair['public_key'];
        $config['secret_key'] = $key_pair['secret_key'];
        break;

      case 'bearer':
        $token = $form_state->getValue('bearer_token');
        if (empty($token)) {
          $form_state->setError($form['bearer_token'], $this->t('Bearer token is required.'));
          return;
        }
        $config['public_key'] = $token;
        $config['secret_key'] = $token;
        break;

      case 'basic':
        $basic_auth = $form_state->getValue('basic_auth');
        if (empty($basic_auth['username']) || empty($basic_auth['password'])) {
          $form_state->setError($form['basic_auth'], $this->t('Both username and password are required.'));
          return;
        }
        $config['public_key'] = $basic_auth['username'];
        $config['secret_key'] = $basic_auth['password'];
        break;
    }

    // Attempt a connection test via the injected LangFuse client.
    if (!$this->langFuseClient->checkConnection($config)) {
      $form_state->setError($form['langfuse_url'], $this->t('Could not connect to LangFuse. Please verify your settings.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('langfuse.settings')
      ->set('langfuse_url', $form_state->getValue('langfuse_url'))
      ->set('auth_method', $form_state->getValue('auth_method'))
      ->set('bearer_token', $form_state->getValue('bearer_token'))
      ->set('basic_auth.username', $form_state->getValue(['basic_auth', 'username']))
      ->set('basic_auth.password', $form_state->getValue(['basic_auth', 'password']))
      ->set('key_pair.public_key', $form_state->getValue(['key_pair', 'public_key']))
      ->set('key_pair.secret_key', $form_state->getValue(['key_pair', 'secret_key']))
      ->set('timeout', $form_state->getValue('timeout'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
