<?php

namespace Drupal\langfuse\Service;

use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\search_api\Query\QueryInterface;
use Dropsolid\LangFuse\Observability\ObservationInterface;

/**
 * Defines the contract for trace management helpers.
 */
interface LangFuseTraceManagerInterface {

  /**
   * Ensures a trace exists, creating it if needed.
   *
   * @param array $metadata
   *   Metadata to include when creating the trace.
   * @param array $tags
   *   Additional tags to append for new traces.
   * @param string|null $initialInput
   *   Optional input to set when the trace is created.
   *
   * @return array{trace:?\Dropsolid\LangFuse\Observability\Trace, created:bool}
   *   The trace object and whether it was created in this call.
   */
  public function ensureTrace(array $metadata = [], array $tags = [], ?string $initialInput = NULL): array;

  /**
   * Registers the parent runner relationship for an agent runner.
   */
  public function registerRunnerParent(string $runnerId, ?string $parentRunnerId): void;

  /**
   * Fetches the registered parent runner for a runner id, if any.
   */
  public function getRunnerParent(string $runnerId): ?string;

  /**
   * Clears the entire runner context (parent, pending, delegation).
   */
  public function clearRunnerContext(string $runnerId): void;

  /**
   * Queues a generation as the parent for upcoming tool spans.
   */
  public function scheduleToolParent(string $runnerId, ObservationInterface $observation): void;

  /**
   * Claims the queued tool parent for a runner.
   */
  public function claimToolParent(string $runnerId): ?ObservationInterface;

  /**
   * Releases the claim on a tool parent.
   */
  public function releaseToolParent(string $runnerId): void;

  /**
   * Starts delegation context for nested agent calls.
   */
  public function startDelegation(string $runnerId, ObservationInterface $span): void;

  /**
   * Ends delegation context for a runner.
   */
  public function endDelegation(string $runnerId): void;

  /**
   * Fetches the active delegation span for a runner.
   */
  public function getDelegationSpan(string $runnerId): ?ObservationInterface;

  /**
   * Enters a named span context (e.g. ai_search.tool).
   */
  public function enterSpanContext(string $context, ObservationInterface $span): void;

  /**
   * Leaves a named span context.
   */
  public function leaveSpanContext(string $context, ObservationInterface $span): void;

  /**
   * Gets the active span for a named context.
   */
  public function currentSpanContext(string $context): ?ObservationInterface;

  /**
   * Determines if a tags array contains a value.
   */
  public function tagsContainValue(array $tags, string $needle): bool;

  /**
   * Normalizes context values for tool spans.
   */
  public function normalizeToolContextValues(ExecutableFunctionCallInterface $tool): array;

  /**
   * Builds metadata for Search API spans.
   */
  public function buildSearchSpanMetadata(QueryInterface $query): array;

  /**
   * Normalizes the Search API query input to a string.
   */
  public function normalizeSearchQueryInput(QueryInterface $query): ?string;

  /**
   * Determines whether the query uses the AI Search backend.
   */
  public function isAiSearchQuery(QueryInterface $query): bool;

  /**
   * Generates a consistent key for Search API spans.
   */
  public function getSearchSpanKey(QueryInterface $query): string;

  /**
   * Creates a safe preview string for metadata.
   */
  public function createSafePreview(mixed $data, int $limit = 400): ?string;

}
