<?php

namespace Drupal\langfuse_example\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\langfuse\LangFuseClientInterface;
use OpenAI\Client as OpenAIClient;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a demo form for OpenAI integration with LangFuse.
 */
class OpenAiDemoForm extends FormBase {

  /**
   * The LangFuse client.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langfuseClient;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new OpenAiDemoForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\langfuse\LangFuseClientInterface $langfuse_client
   *   The LangFuse client.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LangFuseClientInterface $langfuse_client,
    AccountProxyInterface $current_user,
    LoggerChannelFactoryInterface $logger_factory,
  ) {
    $this->configFactory = $config_factory;
    $this->langfuseClient = $langfuse_client;
    $this->currentUser = $current_user;
    $this->logger = $logger_factory->get('langfuse_example');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('langfuse.client'),
      $container->get('current_user'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'langfuse_example_demo_form';
  }

  /**
   * Creates an OpenAI client instance.
   *
   * @return \OpenAI\Client|null
   *   The OpenAI client or NULL if not configured.
   */
  protected function createOpenAiClient(): ?OpenAIClient {
    $config = $this->configFactory->get('langfuse_example.settings');
    $api_key = $config->get('api_key');
    $organization = $config->get('organization');

    if (empty($api_key)) {
      return NULL;
    }

    $factory = \OpenAI::factory()->withApiKey($api_key);

    if (!empty($organization)) {
      $factory = $factory->withOrganization($organization);
    }

    return $factory->make();
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Check if OpenAI API key is configured.
    $config = $this->configFactory->get('langfuse_example.settings');
    $api_key = $config->get('api_key');

    // Add debugging information.
    $this->logger->info('Demo form loaded. API key configured: @configured', [
      '@configured' => !empty($api_key) ? 'Yes' : 'No',
    ]);

    if (empty($api_key)) {
      $form['error'] = [
        '#type' => 'markup',
        '#markup' => $this->t('OpenAI API key is not configured. Please <a href=":url">configure it first</a>.', [
          ':url' => '/admin/config/system/langfuse/openai-settings',
        ]),
      ];
      return $form;
    }

    // Check if LangFuse is properly configured.
    if (!$this->langfuseClient->isConfigured()) {
      $form['langfuse_error'] = [
        '#type' => 'markup',
        '#markup' => $this->t('LangFuse is not properly configured. Please <a href=":url">configure it first</a>.', [
          ':url' => '/admin/config/system/langfuse/settings',
        ]),
      ];
      return $form;
    }

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Enter your prompt'),
      '#description' => $this->t('Type a prompt to send to OpenAI. This will be tracked with LangFuse.'),
      '#default_value' => $form_state->getValue('prompt') ?: 'Explain how LangFuse helps track AI applications in 3 bullet points.',
      '#rows' => 4,
      '#required' => TRUE,
    ];

    $form['model'] = [
      '#type' => 'select',
      '#title' => $this->t('Model'),
      '#options' => [
        'gpt-4o-mini' => $this->t('GPT-4o Mini'),
        'gpt-4.1-mini' => $this->t('GPT-4.1 Mini'),
      ],
      '#default_value' => $form_state->getValue('model') ?: $config->get('model') ?: 'gpt-4o-mini',
    ];

    $form['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#description' => $this->t('Higher values produce more random outputs. Lower values are more deterministic.'),
      '#default_value' => $form_state->getValue('temperature') ?: 0.7,
      '#step' => 0.1,
      '#min' => 0,
      '#max' => 2,
    ];

    // Display results if available.
    $results = $form_state->get('results');
    if ($results) {
      $form['results'] = [
        '#type' => 'details',
        '#title' => $this->t('AI Response'),
        '#open' => TRUE,
      ];

      $form['results']['response'] = [
        '#type' => 'markup',
        '#markup' => '<div class="openai-response"><strong>' . $this->t('Response:') . '</strong><br/>' . nl2br(htmlspecialchars($results['response'])) . '</div>',
      ];

      $form['results']['metadata'] = [
        '#type' => 'details',
        '#title' => $this->t('LangFuse Trace Details'),
        '#collapsed' => TRUE,
      ];

      $form['results']['metadata']['trace_id'] = [
        '#type' => 'markup',
        '#markup' => '<p><strong>' . $this->t('Trace ID:') . '</strong> ' . $results['trace_id'] . '</p>',
      ];

      if (isset($results['usage'])) {
        $form['results']['metadata']['usage'] = [
          '#type' => 'markup',
          '#markup' => '<p><strong>' . $this->t('Token Usage:') . '</strong> ' . json_encode($results['usage']) . '</p>',
        ];
      }
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate Response'),
    ];

    // Add form attributes for styling.
    $form['#attributes']['class'][] = 'openai-demo-form';

    // Add some CSS.
    $form['#attached']['library'][] = 'langfuse_example/langfuse_example.styles';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $prompt = $form_state->getValue('prompt');
    $model = $form_state->getValue('model');
    $temperature = (float) $form_state->getValue('temperature');

    // Create OpenAI client.
    $openai_client = $this->createOpenAiClient();
    if (!$openai_client) {
      $this->messenger()->addError($this->t('OpenAI API key is not configured properly.'));
      return;
    }

    try {
      // Create a simple LangFuse trace for this chat completion.
      $trace = $this->langfuseClient->createTrace(
        'openai_chat_completion',
        (string) $this->currentUser->id(),
        session_id(),
        [
          'form_id' => $this->getFormId(),
          'user_prompt' => $prompt,
        ],
        ['openai', 'demo', 'chat']
      );

      $this->logger->info('Starting OpenAI API call with model: @model, temperature: @temp', [
        '@model' => $model,
        '@temp' => $temperature,
      ]);

      // Create a generation directly in the trace - no manual spans needed.
      $generation = $trace->createGeneration(
        'chat_completion',
        $model,
        [
          'temperature' => $temperature,
          'max_tokens' => 500,
        ],
        ['purpose' => 'demo_chat'],
        [['role' => 'user', 'content' => $prompt]]
      );

      // Track timing for comparison with LangFuse SDK timing.
      $start_time = microtime(TRUE);

      // Make the API request using the OpenAI PHP client.
      $response = $openai_client->chat()->create([
        'model' => $model,
        'messages' => [
          ['role' => 'user', 'content' => $prompt],
        ],
        'temperature' => $temperature,
        'max_tokens' => 500,
      ]);

      $end_time = microtime(TRUE);
      $manual_duration = $end_time - $start_time;

      // Log manual timing for comparison with LangFuse SDK.
      $this->logger->info('Manual timing: OpenAI API call took @duration seconds', [
        '@duration' => round($manual_duration, 3),
      ]);

      // Extract response data with safe property access.
      $completion_text = $response->choices[0]->message->content ?? '';
      $usage = [
        'prompt_tokens' => $response->usage->promptTokens ?? 0,
        'completion_tokens' => $response->usage->completionTokens ?? 0,
        'total_tokens' => $response->usage->totalTokens ?? 0,
      ];

      $this->logger->info('OpenAI API success: generated @chars characters, tokens: @tokens', [
        '@chars' => strlen($completion_text),
        '@tokens' => $usage['total_tokens'],
      ]);

      // End the generation with completion data - SDK handles timing.
      $generation->end([
        'output' => [
          'role' => 'assistant',
          'content' => $completion_text,
        ],
        'usage_details' => $usage,
      ]);

      // End the trace - SDK calculates duration automatically.
      $trace->end();

      $this->logger->info('LangFuse trace completed. Trace ID: @trace_id. Compare manual timing (@manual_duration s) with LangFuse SDK timing in dashboard.', [
        '@trace_id' => $trace->getId(),
        '@manual_duration' => round($manual_duration, 3),
      ]);

      // Store results to display in the form.
      $form_state->set('results', [
        'response' => $completion_text,
        'trace_id' => $trace->getId(),
        'usage' => $usage,
      ]);

      // Rebuild the form to display results.
      $form_state->setRebuild(TRUE);

      $this->logger->info('Demo form completed successfully');
    }
    catch (\Exception $e) {
      $this->logger->error('OpenAI API error: @error', ['@error' => $e->getMessage()]);
      $this->messenger()->addError(
        $this->t('Error: @error', ['@error' => $e->getMessage()])
      );
    }
  }

}
