<?php

namespace Drupal\langfuse;

use Dropsolid\LangFuse\Observability\Trace;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\State\StateInterface;
use Dropsolid\LangFuse\Client as LangFuseSdkClient;

/**
 * The LangFuse client service.
 */
class LangFuseClient implements LangFuseClientInterface {

  /**
   * The LangFuse SDK client instance.
   *
   * @var \Dropsolid\LangFuse\Client|null
   */
  protected ?LangFuseSdkClient $sdkClient = NULL;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * A collection of trace IDs that are waiting to be synced.
   *
   * @var array
   */
  protected array $pendingTraceIds = [];

  /**
   * Constructs a new LangFuseClient.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   */
  public function __construct(
    ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    CacheBackendInterface $cache,
    StateInterface $state,
  ) {
    $this->configFactory = $configFactory;
    $this->logger = $loggerFactory->get('langfuse');
    $this->cache = $cache;
    $this->state = $state;

    // Load any pending trace IDs from the cache.
    $pendingData = $this->cache->get('langfuse_pending_traces');
    if ($pendingData) {
      $this->pendingTraceIds = $pendingData->data;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getSdkClient(): LangFuseSdkClient {
    if ($this->sdkClient === NULL) {
      $this->initializeSdkClient();
    }
    return $this->sdkClient;
  }

  /**
   * Initializes the SDK client based on Drupal configuration.
   *
   * @return \Dropsolid\LangFuse\Client
   *   The initialized SDK client.
   *
   * @throws \Exception
   *   If client initialization fails.
   */
  protected function initializeSdkClient(): LangFuseSdkClient {
    $config = $this->configFactory->get('langfuse.settings');
    $timeout = $config->get('timeout') ?? 10.0;

    $sdkConfig = [
      'host' => $config->get('langfuse_url'),
      'timeout' => $timeout,
    ];

    $authMethod = $config->get('auth_method') ?? 'none';

    if (empty($sdkConfig['host'])) {
      throw new \Exception('LangFuse URL is not configured');
    }

    // Set authentication details based on the selected method.
    switch ($authMethod) {
      case 'bearer':
        $token = $config->get('bearer_token');
        $sdkConfig['public_key'] = $token;
        $sdkConfig['secret_key'] = $token;
        break;

      case 'basic':
        $basicAuth = $config->get('basic_auth');
        $sdkConfig['public_key'] = $basicAuth['username'] ?? '';
        $sdkConfig['secret_key'] = $basicAuth['password'] ?? '';
        break;

      case 'key_pair':
        $keyPair = $config->get('key_pair');
        $sdkConfig['public_key'] = $keyPair['public_key'] ?? '';
        $sdkConfig['secret_key'] = $keyPair['secret_key'] ?? '';
        break;
    }

    try {
      $this->sdkClient = new LangFuseSdkClient($sdkConfig);
      return $this->sdkClient;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to initialize LangFuse client: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createTrace(
    string $name,
    ?string $userId = NULL,
    ?string $sessionId = NULL,
    ?array $metadata = NULL,
    array $tags = [],
    ?string $version = NULL,
    ?string $release = NULL,
  ): Trace {
    try {
      $client = $this->getSdkClient();
      $trace = $client->trace(
        $name,
        $userId,
        $sessionId,
        $metadata,
        $tags,
        $version,
        $release
      );

      // Store the trace ID for later syncing.
      $this->pendingTraceIds[] = $trace->getId();
      $this->cache->set('langfuse_pending_traces', $this->pendingTraceIds);

      // Store the trace object in state.
      $activeTraces = $this->state->get('langfuse_active_traces', []);
      $activeTraces[$trace->getId()] = $trace;
      $this->state->set('langfuse_active_traces', $activeTraces);

      // Set this as the current trace.
      $this->setCurrentTrace($trace->getId());

      return $trace;
    }
    catch (\Exception $e) {
      $this->logger->error('Error creating LangFuse trace: @message', [
        '@message' => $e->getMessage(),
      ]);
      throw new \RuntimeException('Failed to create LangFuse trace', 0, $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function syncTraces(bool $force = FALSE): void {
    if (empty($this->pendingTraceIds) || (!$force && !$this->shouldSyncNow())) {
      return;
    }

    try {
      $client = $this->getSdkClient();
      // The SDK's syncTraces() will handle all traces.
      $client->syncTraces();

      // Clear both pending IDs and active traces.
      $this->pendingTraceIds = [];
      $this->cache->delete('langfuse_pending_traces');
      $this->state->delete('langfuse_active_traces');
      // Use clearCurrentTrace() instead of direct state deletion.
      $this->clearCurrentTrace();

      $this->logger->info('Successfully synced @count LangFuse traces.', [
        '@count' => count($this->pendingTraceIds),
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to sync LangFuse traces: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Determines if traces should be synced at this moment.
   *
   * @return bool
   *   TRUE if traces should be synced now, FALSE otherwise.
   */
  protected function shouldSyncNow(): bool {
    // Check if the request is ending or other conditions that would
    // indicate it's appropriate to sync traces now.
    return php_sapi_name() !== 'cli';
  }

  /**
   * {@inheritdoc}
   */
  public function isConfigured(): bool {
    $config = $this->configFactory->get('langfuse.settings');
    $url = $config->get('langfuse_url');
    $authMethod = $config->get('auth_method');

    if (empty($url)) {
      return FALSE;
    }

    switch ($authMethod) {
      case 'bearer':
        return !empty($config->get('bearer_token'));

      case 'basic':
        return !empty($config->get('basic_auth.username')) && !empty($config->get('basic_auth.password'));

      case 'key_pair':
        return !empty($config->get('key_pair.public_key')) && !empty($config->get('key_pair.secret_key'));

      default:
        return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTrace(string $traceId): ?Trace {
    try {
      // Check if we have the trace in our active traces.
      $activeTraces = $this->state->get('langfuse_active_traces', []);
      if (isset($activeTraces[$traceId])) {
        return $activeTraces[$traceId];
      }

      // If not found in active traces, try to get it from SDK
      // Note: SDK doesn't currently support this, but leaving as placeholder.
      $this->logger->warning('Trace @id not found in active traces', [
        '@id' => $traceId,
      ]);
      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error('Error retrieving LangFuse trace: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentTrace(): ?Trace {
    // Get the current trace from state or session.
    $currentTraceId = $this->state->get('langfuse_current_trace_id');
    if (!$currentTraceId) {
      return NULL;
    }

    return $this->getTrace($currentTraceId);
  }

  /**
   * {@inheritdoc}
   */
  public function setCurrentTrace(string $traceId): void {
    $this->state->set('langfuse_current_trace_id', $traceId);
  }

  /**
   * {@inheritdoc}
   */
  public function clearCurrentTrace(): void {
    $this->state->delete('langfuse_current_trace_id');
    $this->logger->debug('Cleared current trace');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): array {
    $config = $this->configFactory->get('langfuse.settings');
    return [
      'langfuse_url' => $config->get('langfuse_url'),
      'auth_method' => $config->get('auth_method'),
      'bearer_token' => $config->get('bearer_token'),
      'basic_auth' => $config->get('basic_auth'),
      'key_pair' => $config->get('key_pair'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function checkConnection(array $config = []): bool {
    // For testing purposes, allow state to override connection check.
    $testResponse = $this->state->get('langfuse_test_connection_response');
    if ($testResponse !== NULL) {
      return (bool) $testResponse;
    }

    try {
      // If no config provided, use current settings.
      if (empty($config)) {
        $client = $this->getSdkClient();
      }
      else {
        // Create a temporary client with provided config.
        $client = new LangFuseSdkClient($config);
      }

      // Test connection by creating a minimal test trace using proper SDK
      // methods. This tests authentication and network connectivity.
      $testTrace = $client->trace(
        'connection_test_' . time(),
        'system',
        'connection_test_session',
        ['test' => 'connection'],
        ['test', 'connection']
      );

      // End the trace to prepare it for sync.
      $testTrace->end();

      // Actually sync the trace to test server connectivity.
      // This is the critical part that tests actual network connectivity.
      $client->syncTraces();

      // If we get here without exception, connection works.
      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Connection test failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

}
