<?php

declare(strict_types=1);

namespace Drupal\Tests\langfuse\Functional;

use Drupal\Tests\BrowserTestBase;
use Drupal\user\UserInterface;

/**
 * Tests the LangFuse settings form functionality.
 *
 * @group langfuse
 */
class LangFuseSettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'config',
    'langfuse',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The admin user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create and log in as an admin user.
    $this->adminUser = $this->drupalCreateUser([
      'administer site configuration',
      'access administration pages',
    ]);
    $this->drupalLogin($this->adminUser);

    // Bypass any external validation calls.
    $this->config('langfuse.settings')
      ->set('skip_validation', TRUE)
      ->save();

    // Rebuild routes after installing.
    $this->container->get('router.builder')->rebuild();
  }

  /**
   * Tests that form fields render and are conditionally displayed.
   */
  public function testSettingsFormFields(): void {
    // Access the settings form.
    $this->drupalGet('admin/config/system/langfuse/settings');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('LangFuse Settings');

    // Basic form elements.
    $this->assertSession()->fieldExists('edit-langfuse-url');
    $this->assertSession()->fieldExists('edit-auth-method');
    $this->assertSession()->buttonExists('Save configuration');

    // Conditional fields: bearer token.
    $page = $this->getSession()->getPage();
    $page->selectFieldOption('auth_method', 'bearer');
    $this->assertSession()->fieldExists('bearer_token');

    // Conditional fields: key pair.
    $page->selectFieldOption('auth_method', 'key_pair');
    $this->assertSession()->fieldExists('key_pair[public_key]');
    $this->assertSession()->fieldExists('key_pair[secret_key]');
  }

  /**
   * Tests form validation logic (e.g., required fields).
   */
  public function testFormValidation(): void {
    // IMPORTANT FIX: First disable skip_validation to ensure validation runs.
    $this->config('langfuse.settings')
      ->set('skip_validation', FALSE)
      ->save();

    $this->drupalGet('admin/config/system/langfuse/settings');
    $this->assertSession()->statusCodeEquals(200);

    // If the user selects key_pair but leaves keys empty, it should fail.
    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'key_pair',
      'key_pair[public_key]' => '',
      'key_pair[secret_key]' => '',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Both Public and Secret keys are required');

    // Test non-URL input for langfuse_url.
    $edit = [
      'langfuse_url' => 'not-a-url',
      'auth_method' => 'none',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The URL not-a-url is not valid');
  }

  /**
   * Tests saving valid configuration without real connection validation.
   */
  public function testConfigurationSaving(): void {
    $this->drupalGet('admin/config/system/langfuse/settings');

    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'key_pair',
      'key_pair[public_key]' => 'test-public-key',
      'key_pair[secret_key]' => 'test-secret-key',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved');

    // Verify config was updated.
    $config = $this->config('langfuse.settings');
    $this->assertEquals('https://cloud.langfuse.com', $config->get('langfuse_url'));
    $this->assertEquals('key_pair', $config->get('auth_method'));
    $this->assertEquals('test-public-key', $config->get('key_pair.public_key'));
    $this->assertEquals('test-secret-key', $config->get('key_pair.secret_key'));
  }

  /**
   * Tests form validation with skip_validation enabled.
   */
  public function testFormWithSkippedValidation(): void {
    // Ensure validation is skipped.
    $this->config('langfuse.settings')
      ->set('skip_validation', TRUE)
      ->save();

    $this->drupalGet('admin/config/system/langfuse/settings');
    $this->assertSession()->statusCodeEquals(200);

    // Test that we can save incomplete key_pair without validation.
    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'key_pair',
    // Intentionally empty.
      'key_pair[public_key]' => '',
    // Intentionally empty.
      'key_pair[secret_key]' => '',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved');

    // Verify config was saved despite being incomplete.
    $config = $this->config('langfuse.settings');
    $this->assertEquals('key_pair', $config->get('auth_method'));
    $this->assertEmpty($config->get('key_pair.public_key'));
    $this->assertEmpty($config->get('key_pair.secret_key'));
  }

  /**
   * Tests form validation with validation enabled.
   */
  public function testFormWithValidation(): void {
    // Ensure validation is enabled.
    $this->config('langfuse.settings')
      ->set('skip_validation', FALSE)
      ->save();

    $this->drupalGet('admin/config/system/langfuse/settings');
    $this->assertSession()->statusCodeEquals(200);

    // Test key_pair validation.
    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'key_pair',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Both Public and Secret keys are required');

    // Test bearer token validation.
    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'bearer',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Bearer token is required');

    // Test basic auth validation.
    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'basic',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('Both username and password are required');

    // Test URL validation.
    $edit = [
      'langfuse_url' => 'not-a-url',
      'auth_method' => 'none',
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The URL not-a-url is not valid');
  }

  /**
   * Tests saving valid configuration with validation enabled.
   */
  public function testValidConfigurationSaving(): void {
    // Ensure validation is enabled.
    $this->config('langfuse.settings')
      ->set('skip_validation', FALSE)
      ->save();

    // IMPORTANT FIX: Mock the connection check by replacing the service
    // with a test version that always returns true.
    $this->container->get('state')
      ->set('langfuse_test_connection_response', TRUE);

    $this->drupalGet('admin/config/system/langfuse/settings');

    $edit = [
      'langfuse_url' => 'https://cloud.langfuse.com',
      'auth_method' => 'key_pair',
      'key_pair[public_key]' => 'test-public-key',
      'key_pair[secret_key]' => 'test-secret-key',
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved');

    // Verify config was updated.
    $config = $this->config('langfuse.settings');
    $this->assertEquals('https://cloud.langfuse.com', $config->get('langfuse_url'));
    $this->assertEquals('key_pair', $config->get('auth_method'));
    $this->assertEquals('test-public-key', $config->get('key_pair.public_key'));
    $this->assertEquals('test-secret-key', $config->get('key_pair.secret_key'));
  }

}
