<?php

namespace Drupal\Tests\langfuse\Integration;

use Drupal\KernelTests\KernelTestBase;
use Drupal\langfuse\LangFuseClientInterface;

/**
 * Tests the LangFuse client integration with a LangFuse server.
 *
 * These tests validate that the Drupal integration layer works correctly with
 * the LangFuse SDK by creating traces and observations and verifying that
 * they're properly synchronized to a LangFuse server.
 *
 * @group langfuse
 */
class LangFuseClientIntegrationTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'langfuse',
  ];

  /**
   * The LangFuse client service.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langFuseClient;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install necessary schemas.
    $this->installEntitySchema('user');
    $this->installSchema('system', ['sequences']);
    $this->installConfig('langfuse');

    // Configure the LangFuse client for testing.
    $this->config('langfuse.settings')
      ->set('langfuse_url', 'http://langfuse-web:3000')
      ->set('auth_method', 'key_pair')
      ->set('key_pair', [
        'public_key' => 'public-key-placeholder',
        'secret_key' => 'secret-key-placeholder',
      ])
      ->set('skip_validation', TRUE)
      ->save();

    // Get the LangFuse client service.
    $this->langFuseClient = $this->container->get('langfuse.client');

    // Check if LangFuse server is available before running tests.
    $this->checkServerAvailability();
  }

  /**
   * Check if the LangFuse server is available.
   *
   * @throws \PHPUnit\Framework\SkippedTestError
   *   If the LangFuse server is not available.
   *
   * @phpstan-ignore-next-line
   */
  protected function checkServerAvailability(): void {
    try {
      // Use the client's built-in connection check.
      if (!$this->langFuseClient->checkConnection()) {
        $this->markTestSkipped("LangFuse server connection failed. Is the server running at langfuse-web:3000?");
      }
    }
    catch (\Exception $e) {
      $this->markTestSkipped("LangFuse server not available: " . $e->getMessage());
    }
  }

  /**
   * Tests basic trace creation and synchronization.
   */
  public function testBasicTraceCreation(): void {
    $this->checkServerAvailability();

    // 1. Create a simple trace.
    $traceName = 'Drupal Integration Test - ' . uniqid();
    $trace = $this->langFuseClient->createTrace(
      $traceName,
      'test-user-id',
      'test-session-id',
      ['source' => 'drupal-test', 'test_type' => 'basic'],
      ['test', 'integration']
    );
    // 100ms delay
    usleep(100000);

    // Verify trace was created locally.
    $this->assertNotNull($trace, 'Trace should be created locally.');
    $this->assertEquals($traceName, $trace->getName(), 'Trace name should match.');

    // Get the trace ID.
    $traceId = $trace->getId();

    // Add a simple event to the trace.
    $trace->createEvent(
      'Test Event',
      'DEFAULT',
      ['timestamp' => time()]
    );
    // 150ms delay
    usleep(150000);

    // 2. Force synchronization to the server.
    $this->langFuseClient->syncTraces(TRUE);

    // 3. Wait briefly to ensure sync completes
    sleep(1);

    // We confirm the trace is no longer in local state which indicates
    // it was synchronized.
    $retrievedTrace = $this->langFuseClient->getTrace($traceId);
    $this->assertNull($retrievedTrace, 'Trace should be cleared from state after sync');
  }

  /**
   * Tests a complete LangFuse workflow with all observation types.
   *
   * This test creates a complex trace with nested spans, generations,
   * events, and scores to simulate a realistic AI processing pipeline.
   */
  public function testCompleteWorkflow(): void {
    $this->checkServerAvailability();

    // 1. Create a complex trace simulating a RAG process.
    $traceName = 'Drupal RAG Workflow - ' . uniqid();
    $trace = $this->langFuseClient->createTrace(
      $traceName,
      'drupal-user-123',
      'rag-session-456',
      [
        'workflow' => 'complete-test',
        'source' => 'drupal-module',
        'interface' => 'web',
      ],
      ['drupal', 'rag', 'test']
    );
    // 100ms delay
    usleep(100000);

    $traceId = $trace->getId();

    // 2. Add an initial event to the trace.
    $trace->createEvent(
      'Query Received',
      'DEFAULT',
      [
        'action' => 'user_question',
        'query' => 'What are the configuration options for Drupal LangFuse?',
      ]
    );
    // 150ms delay
    usleep(150000);

    // 3. Create a query processing span.
    $querySpan = $trace->createSpan(
      'Query Processing',
      [
        'phase' => 'understanding',
        'purpose' => 'query_enhancement',
      ]
    );
    // 100ms delay
    usleep(100000);

    // 4. Create a generation within the query span.
    $queryGen = $querySpan->createGeneration(
      'Query Enhancement',
      'gpt-4',
      ['temperature' => 0.3, 'top_p' => 0.95],
      ['purpose' => 'query_expansion'],
      [
        ['role' => 'system', 'content' => 'Extract search keywords from this query.'],
        ['role' => 'user', 'content' => 'What are the configuration options for Drupal LangFuse?'],
      ]
    );
    // 300ms delay - simulate LLM processing
    usleep(300000);

    // Complete the query generation.
    $queryGen->end([
      'output' => 'drupal langfuse configuration options settings parameters',
      'usage_details' => [
        'prompt_tokens' => 25,
        'completion_tokens' => 6,
        'total_tokens' => 31,
      ],
    ]);
    // 100ms delay
    usleep(100000);
    $querySpan->end(['status' => 'completed']);
    // 150ms delay
    usleep(150000);

    // 5. Create a document retrieval span.
    $retrievalSpan = $trace->createSpan(
      'Document Retrieval',
      [
        'phase' => 'retrieval',
        'database' => 'vector_db',
      ]
    );
    // 200ms delay - simulate search
    usleep(200000);

    // Add document retrieval event.
    $retrievalSpan->createEvent(
      'Documents Retrieved',
      'DEFAULT',
      [
        'document_count' => 3,
        'search_terms' => ['drupal', 'langfuse', 'configuration'],
      ]
    );
    // 150ms delay
    usleep(150000);
    $retrievalSpan->end(['status' => 'completed']);
    // 100ms delay
    usleep(100000);

    // 6. Create a final response generation.
    $finalGen = $trace->createGeneration(
      'Response Generation',
      'gpt-4-turbo',
      ['temperature' => 0.7, 'max_tokens' => 500],
      ['purpose' => 'final_response'],
      [
        ['role' => 'system', 'content' => 'You are a helpful assistant for Drupal users.'],
        ['role' => 'user', 'content' => 'What are the configuration options for Drupal LangFuse?'],
        ['role' => 'assistant', 'content' => 'I found several configuration options for the Drupal LangFuse module:'],
      ]
    );
    // 400ms delay - simulate final LLM processing
    usleep(400000);

    // Complete the final generation.
    $finalGen->end([
      'output' => 'The Drupal LangFuse module offers several configuration options including: API key settings, connection URL, authentication method (bearer token, basic auth, or key pair), timeout settings, and more. These can be configured at /admin/config/system/langfuse/settings.',
      'usage_details' => [
        'prompt_tokens' => 120,
        'completion_tokens' => 45,
        'total_tokens' => 165,
      ],
    ]);
    // 150ms delay
    usleep(150000);

    // 7. Add scores to various components.
    $trace->score('query_understanding', 0.92, 'Accurate keyword extraction', $queryGen);
    // 100ms delay
    usleep(100000);
    $trace->score('retrieval_quality', 0.85, 'Good document retrieval', $retrievalSpan);
    // 100ms delay
    usleep(100000);
    $trace->score('response_quality', 0.95, 'Comprehensive and accurate response', $finalGen);
    // 100ms delay
    usleep(100000);

    // 8. End the trace
    $trace->end(['status' => 'completed']);
    // 100ms delay
    usleep(100000);

    // 9. Sync traces to the server.
    $this->langFuseClient->syncTraces(TRUE);

    // Wait briefly to ensure sync completes.
    sleep(1);

    // Verify the trace was properly sent by confirming it's
    // no longer in local state.
    $currentTraceState = $this->langFuseClient->getTrace($traceId);
    $this->assertNull($currentTraceState, 'Trace should be cleared from state after sync');
  }

  /**
   * Tests saving and retrieving traces via the Drupal client.
   *
   * This test verifies the lifecycle management functionality of our client,
   * including setting/getting current traces and proper state management.
   */
  public function testTraceLifecycle(): void {
    $this->checkServerAvailability();

    // 1. Create a trace.
    $trace = $this->langFuseClient->createTrace(
      'Lifecycle Test',
      'lifecycle-user',
      'lifecycle-session',
      ['test_type' => 'lifecycle'],
      ['lifecycle', 'test']
    );
    $traceId = $trace->getId();
    // 100ms delay
    usleep(100000);

    // 2. Set as current trace.
    $this->langFuseClient->setCurrentTrace($traceId);
    // 50ms delay
    usleep(50000);

    // 3. Get current trace and verify.
    $currentTrace = $this->langFuseClient->getCurrentTrace();
    $this->assertNotNull($currentTrace);
    $this->assertEquals($traceId, $currentTrace->getId());
    // 50ms delay
    usleep(50000);

    // 4. Add an event.
    $currentTrace->createEvent(
      'Lifecycle Event',
      'DEFAULT',
      ['timestamp' => time()]
    );
    // 100ms delay
    usleep(100000);

    // 5. Add a span with completion
    $span = $currentTrace->createSpan('Test Operation');
    // 150ms delay
    usleep(150000);
    $span->end(['status' => 'success']);
    // 100ms delay
    usleep(100000);

    // 6. Clear current trace.
    $this->langFuseClient->clearCurrentTrace();
    $this->assertNull($this->langFuseClient->getCurrentTrace(), 'Current trace should be null after clear');
    // 50ms delay
    usleep(50000);

    // 7. Retrieve trace by ID.
    $retrievedTrace = $this->langFuseClient->getTrace($traceId);
    $this->assertNotNull($retrievedTrace);
    $this->assertEquals($traceId, $retrievedTrace->getId());
    // 50ms delay
    usleep(50000);

    // 8. End the trace
    $retrievedTrace->end();
    // 50ms delay
    usleep(50000);

    // 9. Sync and verify state changes
    $this->langFuseClient->syncTraces(TRUE);
    // Wait for sync to complete.
    sleep(1);

    // After sync, the trace should no longer be in local state.
    $retrievedTraceAfterSync = $this->langFuseClient->getTrace($traceId);
    $this->assertNull($retrievedTraceAfterSync, 'Trace should be cleared from state after sync');
  }

}
