<?php

declare(strict_types=1);

namespace Drupal\Tests\langfuse\Unit;

use Drupal\Tests\UnitTestCase;
use Drupal\langfuse\LangFuseClient;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\State\StateInterface;
use Prophecy\PhpUnit\ProphecyTrait;

/**
 * Unit tests for LangFuseClient.
 *
 * @group langfuse
 * @coversDefaultClass \Drupal\langfuse\LangFuseClient
 */
class LangFuseClientTest extends UnitTestCase {

  use ProphecyTrait;

  /**
   * The LangFuse client under test.
   *
   * @var \Drupal\langfuse\LangFuseClient
   */
  protected $client;

  /**
   * The mocked Config Factory.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $configFactory;

  /**
   * The mocked Logger Factory.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $loggerFactory;

  /**
   * The mocked Cache Backend.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $cache;

  /**
   * The mocked State service.
   *
   * @var \Prophecy\Prophecy\ObjectProphecy
   */
  protected $state;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Use prophecy for creating mock objects.
    $this->configFactory = $this->prophesize(ConfigFactoryInterface::class);
    $this->loggerFactory = $this->prophesize(LoggerChannelFactoryInterface::class);
    $this->cache = $this->prophesize(CacheBackendInterface::class);
    $this->state = $this->prophesize(StateInterface::class);

    // Mock a logger channel.
    $logger = $this->prophesize(LoggerChannelInterface::class);
    $this->loggerFactory->get('langfuse')->willReturn($logger->reveal());

    // Initialize the client with mocked dependencies.
    $this->client = new LangFuseClient(
      $this->configFactory->reveal(),
      $this->loggerFactory->reveal(),
      $this->cache->reveal(),
      $this->state->reveal()
    );
  }

  /**
   * Tests the isConfigured method to ensure it detects required configs.
   *
   * @covers ::isConfigured
   */
  public function testIsConfigured(): void {
    $config = $this->prophesize(ImmutableConfig::class);

    // Test unconfigured state.
    $config->get('langfuse_url')->willReturn(NULL);
    $config->get('auth_method')->willReturn('none');
    $this->configFactory->get('langfuse.settings')->willReturn($config->reveal());
    $this->assertFalse($this->client->isConfigured(), 'Client is not configured when URL is null.');

    // Test configured state with key pair.
    $config->get('langfuse_url')->willReturn('https://cloud.langfuse.com');
    $config->get('auth_method')->willReturn('key_pair');
    $config->get('key_pair.public_key')->willReturn('test-key');
    $config->get('key_pair.secret_key')->willReturn('test-secret');
    $this->configFactory->get('langfuse.settings')->willReturn($config->reveal());
    $this->assertTrue($this->client->isConfigured(), 'Client is configured with key_pair auth.');
  }

}
