<?php

namespace Drupal\langfuse_ai_agents_logging\EventSubscriber;

use Drupal\ai_agents\Event\AgentFinishedExecutionEvent;
use Drupal\ai_agents\Event\AgentStartedExecutionEvent;
use Drupal\langfuse\Service\LangFuseTraceManagerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Tracks AI agent runner hierarchies for LangFuse logging.
 */
final class LangFuseAgentLifecycleSubscriber implements EventSubscriberInterface {

  /**
   * Constructs the subscriber.
   */
  public function __construct(protected LangFuseTraceManagerInterface $traceManager) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      AgentStartedExecutionEvent::EVENT_NAME => ['onAgentStarted', 0],
      AgentFinishedExecutionEvent::EVENT_NAME => ['onAgentFinished', 0],
    ];
  }

  /**
   * Records the parent runner relationship when an agent starts.
   */
  public function onAgentStarted(AgentStartedExecutionEvent $event): void {
    // Track parent-child relationships between runners so delegated agents can
    // inherit the correct parent span.
    $this->traceManager->registerRunnerParent($event->getAgentRunnerId(), $event->getCallerId());
  }

  /**
   * Clears runner hierarchy context once an agent finishes.
   */
  public function onAgentFinished(AgentFinishedExecutionEvent $event): void {
    $this->traceManager->clearRunnerContext($event->getAgentRunnerId());
  }

}
