<?php

namespace Drupal\langfuse_ai_agents_logging\EventSubscriber;

use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\ai\Service\FunctionCalling\StructuredExecutableFunctionCallInterface;
use Drupal\ai_agents\Event\AgentToolBase;
use Drupal\ai_agents\Event\AgentToolFinishedExecutionEvent;
use Drupal\ai_agents\Event\AgentToolPreExecuteEvent;
use Drupal\ai_agents\PluginInterfaces\AiAgentFunctionInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\langfuse\LangFuseClientInterface;
use Drupal\langfuse\Service\LangFuseTraceManagerInterface;
use Dropsolid\LangFuse\Observability\ObservationInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to AI tool events to create LangFuse spans.
 */
final class LangFuseToolSpanSubscriber implements EventSubscriberInterface {

  private const CONTEXT_SEARCH_TOOL = 'ai_search.tool';

  /**
   * Active tool spans keyed by tool identifier.
   *
   * @var array<string,array{span:\Dropsolid\LangFuse\Observability\ObservationInterface, started:float}>
   */
  private array $activeToolSpans = [];

  /**
   * Runner IDs for claimed tool parents keyed by tool span key.
   *
   * @var array<string,string|null>
   */
  private array $toolParentClaims = [];

  /**
   * Constructs the subscriber.
   */
  public function __construct(
    protected LangFuseClientInterface $langFuseClient,
    protected LangFuseTraceManagerInterface $traceManager,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      AgentToolPreExecuteEvent::EVENT_NAME => ['onAgentToolPreExecute', 0],
      AgentToolFinishedExecutionEvent::EVENT_NAME => ['onAgentToolFinished', 0],
    ];
  }

  /**
   * Starts a span for the tool execution.
   */
  public function onAgentToolPreExecute(AgentToolPreExecuteEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    // Every tool span is part of the long-running Drupal request trace. Create
    // one if missing so the span chain stays continuous.
    $tool = $event->getTool();
    ['trace' => $trace] = $this->traceManager->ensureTrace([
      'first_operation_type' => 'tool_execution',
      'first_tool_plugin' => $tool->getPluginId(),
    ], ['tool']);
    if (!$trace) {
      return;
    }

    $spanName = sprintf('tool:%s', $this->getToolSpanName($tool));
    $contextSnapshot = $this->traceManager->normalizeToolContextValues($tool);

    // If the generation scheduled itself as parent (tool call), claim it now.
    $parentSpan = $this->traceManager->claimToolParent($event->getAgentRunnerId());

    $metadata = array_filter([
      'agent_id' => $event->getAgentId(),
      'agent_runner_id' => $event->getAgentRunnerId(),
      'tool_plugin_id' => $tool->getPluginId(),
      'tool_function_name' => $tool->getFunctionName(),
      'tool_id' => $event->getToolId(),
      'thread_id' => $event->getThreadId(),
      'caller_id' => $event->getCallerId(),
      'progress_message' => $event->getProgressMessage(),
      'context_preview' => $this->traceManager->createSafePreview($contextSnapshot),
      'runner_parent' => $event->getCallerId(),
    ], static fn($value) => $value !== NULL && $value !== '');

    $span = $trace->createSpan($spanName, $metadata ?: NULL, $contextSnapshot ?: NULL);
    if ($parentSpan instanceof ObservationInterface) {
      $span->setParentObservationId($parentSpan->getId());
    }

    $isAiSearchTool = $this->isAiSearchTool($tool);
    if ($tool instanceof AiAgentFunctionInterface) {
      $this->traceManager->startDelegation($event->getAgentRunnerId(), $span);
    }
    if ($isAiSearchTool) {
      $this->traceManager->enterSpanContext(self::CONTEXT_SEARCH_TOOL, $span);
    }

    $key = $this->getToolSpanKey($event);
    $this->toolParentClaims[$key] = $parentSpan ? $event->getAgentRunnerId() : NULL;
    $this->activeToolSpans[$key] = [
      'span' => $span,
      'started' => microtime(TRUE),
    ];
  }

  /**
   * Completes the span when the tool finishes.
   */
  public function onAgentToolFinished(AgentToolFinishedExecutionEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    $key = $this->getToolSpanKey($event);
    if (empty($this->activeToolSpans[$key])) {
      return;
    }
    $span = $this->activeToolSpans[$key]['span'];
    $started = $this->activeToolSpans[$key]['started'];
    unset($this->activeToolSpans[$key]);
    $claimedRunnerId = $this->toolParentClaims[$key] ?? NULL;
    unset($this->toolParentClaims[$key]);
    $tool = $event->getTool();

    $metadata = array_filter([
      'status' => 'completed',
      'finished_timestamp' => $this->time->getRequestTime(),
      'duration_ms' => round(max(0, (microtime(TRUE) - $started) * 1000), 2),
      'progress_message' => $event->getProgressMessage(),
    ], static fn($value) => $value !== NULL && $value !== '');

    $outputPayload = [
      'readable' => $tool->getReadableOutput(),
    ];

    if ($tool instanceof StructuredExecutableFunctionCallInterface) {
      $structuredOutput = $tool->getStructuredOutput();
      if (!empty($structuredOutput)) {
        $outputPayload['structured'] = $structuredOutput;
        $metadata['structured_output_preview'] = $this->traceManager->createSafePreview($structuredOutput);
      }
    }

    $span->end([
      'metadata' => $metadata,
      'output' => $outputPayload,
    ]);

    $isAiSearchTool = $this->isAiSearchTool($tool);
    if ($claimedRunnerId) {
      // Generation claims are reference-counted. Release once the tool finishes.
      $this->traceManager->releaseToolParent($claimedRunnerId);
    }

    if ($tool instanceof AiAgentFunctionInterface) {
      $this->traceManager->endDelegation($event->getAgentRunnerId());
    }
    if ($isAiSearchTool) {
      $this->traceManager->leaveSpanContext(self::CONTEXT_SEARCH_TOOL, $span);
    }
  }

  /**
   * Generates a consistent key for tracked tool spans.
   */
  private function getToolSpanKey(AgentToolBase $event): string {
    return $event->getToolId() ?: (string) spl_object_id($event->getTool());
  }

  /**
   * Builds a readable name for tool spans.
   */
  private function getToolSpanName(ExecutableFunctionCallInterface $tool): string {
    return $tool->getFunctionName() ?: $tool->getPluginId();
  }

  /**
   * Determines whether the tool represents the AI Search RAG tool.
   */
  private function isAiSearchTool(ExecutableFunctionCallInterface $tool): bool {
    return $tool->getPluginId() === 'ai_search_rag_search'
      || $tool->getFunctionName() === 'ai_search_rag_search';
  }

}
