<?php

namespace Drupal\langfuse_ai_logging\EventSubscriber;

use Drupal\ai\Event\AiProviderResponseBaseEvent;
use Drupal\ai\Event\PostGenerateResponseEvent;
use Drupal\ai\Event\PostStreamingResponseEvent;
use Drupal\ai\Event\PreGenerateResponseEvent;
use Drupal\ai\OperationType\Chat\StreamedChatMessageIteratorInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutputInterface;
use Drupal\ai\OperationType\Chat\Tools\ToolsPropertyResultInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\langfuse\LangFuseClientInterface;
use Drupal\langfuse\Service\LangFuseTraceManagerInterface;
use Dropsolid\LangFuse\Observability\ObservationInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Event subscriber for comprehensive AI interaction tracing with LangFuse.
 *
 * This subscriber implements the proper LangFuse integration pattern:
 * 1. Pre-generation: Create trace when AI request starts
 * 2. Post-generation: End trace and capture response data
 * 3. Streaming: Handle streaming responses appropriately
 * Uses singleton pattern via LangFuse client's current trace management.
 */
class LangFuseAiLoggingSubscriber implements EventSubscriberInterface {

  /**
   * The LangFuse client service.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langFuseClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface
   */
  protected TimeInterface $time;

  /**
   * The trace manager service.
   *
   * @var \Drupal\langfuse\Service\LangFuseTraceManagerInterface
   */
  protected LangFuseTraceManagerInterface $traceManager;

  /**
   * Active generations keyed by request thread ID.
   *
   * @var array<string,\Dropsolid\LangFuse\Observability\ObservationInterface>
   */
  protected array $activeGenerations = [];

  /**
   * Constructs a new LangFuseAiLoggingSubscriber.
   *
   * @param \Drupal\langfuse\LangFuseClientInterface $langfuse_client
   *   The LangFuse client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\langfuse\Service\LangFuseTraceManagerInterface $trace_manager
   *   The trace manager service.
   */
  public function __construct(
    LangFuseClientInterface $langfuse_client,
    LoggerChannelFactoryInterface $logger_factory,
    TimeInterface $time,
    LangFuseTraceManagerInterface $trace_manager,
  ) {
    $this->langFuseClient = $langfuse_client;
    $this->logger = $logger_factory->get('langfuse_ai_logging');
    $this->time = $time;
    $this->traceManager = $trace_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      PreGenerateResponseEvent::EVENT_NAME => ['onPreGenerateResponse', 0],
      PostGenerateResponseEvent::EVENT_NAME => ['onPostGenerateResponse', 0],
      PostStreamingResponseEvent::EVENT_NAME => ['onPostStreamingResponse', 0],
    ];
  }

  /**
   * Handles the pre-generate response event.
   *
   * Creates a LangFuse trace when AI request starts, or reuses existing trace
   * if one already exists for this Drupal request.
   *
   * @param \Drupal\ai\Event\PreGenerateResponseEvent $event
   *   The pre-generation event.
   */
  public function onPreGenerateResponse(PreGenerateResponseEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      $this->logger->debug('LangFuse not configured, skipping AI request tracing');
      return;
    }

    try {
      $threadId = $event->getRequestThreadId();
      $providerId = $event->getProviderId();
      $operationType = $event->getOperationType();
      $modelId = $event->getModelId();
      $runnerId = $this->extractRunnerIdFromTags($event->getTags());
      $parentRunnerId = $runnerId ? $this->traceManager->getRunnerParent($runnerId) : NULL;

      // Create or reuse trace for this request.
      ['trace' => $trace, 'created' => $traceCreated] = $this->traceManager->ensureTrace([
        'first_ai_provider' => $providerId,
        'first_operation_type' => $operationType,
      ], ['multi_operation'], $event->getInput() ? $event->getInput()->toString() : NULL);

      if (!$trace) {
        return;
      }

      // Create generation metadata with AI-specific details.
      $generationMetadata = [
        'provider_id' => json_encode($providerId),
        'operation_type' => json_encode($operationType),
        'model_id' => json_encode($modelId),
        'configuration_preview' => $this->traceManager->createSafePreview(json_encode($event->getConfiguration())),
        'input_preview' => $this->traceManager->createSafePreview($event->getInput() ? $event->getInput()->toString() : ''),
        'drupal_request_time' => json_encode($this->time->getRequestTime()),
        'is_embedding_operation' => json_encode(in_array($operationType, ['embedding', 'embeddings'])),
        'thread_id' => json_encode($threadId),
      ];

      if ($runnerId) {
        $generationMetadata['runner_id'] = json_encode($runnerId);
      }

      // Add input metadata if available.
      if ($event->getInput()) {
        $generationMetadata['input_type'] = get_class($event->getInput());
        $inputString = $event->getInput()->toString();
        $generationMetadata['input_preview'] = $this->traceManager->createSafePreview($inputString);
        $generationMetadata['input_length'] = mb_strlen($inputString);
      }

      // Create generation within the shared trace.
      $generation_name = match($operationType) {
        'embedding', 'embeddings' => 'text-embedding',
        'chat_completion', 'completion' => 'chat-completion',
        'text_completion' => 'text-completion',
        default => sprintf('%s_%s', $operationType, $providerId),
      };

      // Ensure model parameters are in correct format (associative array).
      $modelParameters = $event->getConfiguration();

      // Debug: Log what we're getting from the AI module.
      $this->logger->debug('Model parameters from AI module: @params', [
        '@params' => json_encode($modelParameters),
      ]);

      if (is_array($modelParameters) && !empty($modelParameters)) {
        // Ensure it's a proper associative array for LangFuse API.
        // If it's a list (indexed array), convert to empty array.
        if (array_is_list($modelParameters)) {
          $this->logger->warning('Model parameters are indexed array, converting to empty array for LangFuse');
          $modelParameters = [];
        }
      }
      else {
        // Fallback to empty array if no configuration.
        $modelParameters = [];
      }

      $generation = $trace->createGeneration(
        $generation_name,
        $modelId,
        $modelParameters,
        $generationMetadata,
        $event->getInput() ? $event->getInput()->toString() : NULL
      );

      // Prefer the span registered through agent delegation (nested agents).
      $parentSpan = NULL;
      if ($parentRunnerId) {
        $parentSpan = $this->traceManager->getDelegationSpan($parentRunnerId);
      }
      // Embeddings originate from Search API; attach to active retrieval spans.
      if (!$parentSpan && $operationType === 'embeddings') {
        $parentSpan = $this->traceManager->currentSpanContext('ai_search.retrieval')
          ?? $this->traceManager->currentSpanContext('ai_search.tool');
      }
      if ($parentSpan instanceof ObservationInterface) {
        $generation->setParentObservationId($parentSpan->getId());
      }

      $this->logger->info('Created generation for @operation_type via @provider in trace @trace_id (thread: @thread_id)', [
        '@operation_type' => $operationType,
        '@provider' => $providerId,
        '@trace_id' => $trace->getId(),
        '@thread_id' => $threadId,
      ]);

      $this->activeGenerations[$threadId] = $generation;
    }
    catch (\Exception $e) {
      // Log error but don't interfere with AI request.
      $this->logger->error('Failed to create LangFuse trace for AI request: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Handles the post-generate response event.
   *
   * Completes the generation within the shared trace and captures response
   * data. Only ends the trace if this is the last AI operation in the request.
   *
   * @param \Drupal\ai\Event\PostGenerateResponseEvent $event
   *   The post-generation event.
   */
  public function onPostGenerateResponse(PostGenerateResponseEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    $trace = $this->langFuseClient->getCurrentTrace();
    if (!$trace) {
      $this->logger->warning('No LangFuse trace available for post-generation handling');
      return;
    }

    try {
      // Defer streamed responses until the post-streaming event.
      if ($this->isStreamingOutput($event)) {
        return;
      }

      // Complete the generation with response data.
      $completed = $this->completeGeneration($trace, $event);

      if (!$completed) {
        return;
      }

      // Update trace metadata with latest operation info.
      $trace->updateMetadata([
        'last_response_timestamp' => time(),
        'total_operations' => count($trace->getObservations()),
      ]);
      // Note: We don't end the trace here as other AI operations might follow.
      // The trace will be ended when the Drupal request finishes.
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to complete LangFuse generation for AI response: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Handles the post-streaming response event.
   *
   * Captures final streaming results and associates with trace.
   *
   * @param \Drupal\ai\Event\PostStreamingResponseEvent $event
   *   The post-streaming event.
   */
  public function onPostStreamingResponse(PostStreamingResponseEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    $trace = $this->langFuseClient->getCurrentTrace();
    if (!$trace) {
      $this->logger->debug('No LangFuse trace available for streaming completion');
      return;
    }

    try {
      // Complete the deferred generation with the finalized streamed output.
      $completed = $this->completeGeneration($trace, $event);

      if (!$completed) {
        $this->logger->warning('Streaming completion event triggered without finalized output for thread @thread_id', [
          '@thread_id' => $event->getRequestThreadId(),
        ]);
        return;
      }

      // Update trace metadata with latest operation info.
      $trace->updateMetadata([
        'last_response_timestamp' => time(),
        'total_operations' => count($trace->getObservations()),
      ]);

      // Update trace with streaming completion metadata.
      $this->finalizeTrace($trace, $event, 'streaming');

    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update LangFuse trace for streaming response: @error', [
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Completes the generation within a trace.
   *
   * @param mixed $trace
   *   The trace object.
   * @param \Drupal\ai\Event\AiProviderResponseBaseEvent $event
   *   The AI provider response event.
   *
   * @return bool
   *   TRUE if the generation was completed, FALSE if it was deferred.
   */
  private function completeGeneration($trace, AiProviderResponseBaseEvent $event): bool {
    $targetThreadId = $event->getRequestThreadId();
    if (empty($this->activeGenerations[$targetThreadId])) {
      $this->logger->warning('No matching generation found for thread @thread_id', [
        '@thread_id' => $targetThreadId,
      ]);
      return FALSE;
    }
    $generation = $this->activeGenerations[$targetThreadId];
    $normalizedTools = [];

    // Prepare output data - trust the AI module's normalization.
    $outputData = [];
    $output = $event->getOutput();
    $generationMetadata = $generation->getMetadata() ?? [];
    $runnerId = NULL;
    if (isset($generationMetadata['runner_id'])) {
      $runnerId = json_decode($generationMetadata['runner_id'], TRUE);
    }

    try {
      $chatMessage = NULL;

      if (method_exists($output, 'getNormalized')) {
        $chatMessage = $output->getNormalized();
      }

      if ($chatMessage instanceof StreamedChatMessageIteratorInterface) {
        return FALSE;
      }

      // Handle different output types safely.
      $extraMetadata = [];
      if ($chatMessage instanceof ChatMessage && $chatMessage->getTools()) {
        $normalizedTools = $this->normalizeToolCalls($chatMessage->getTools());
        if (!empty($normalizedTools)) {
          $extraMetadata['tool_calls'] = $normalizedTools;
          if ($runnerId) {
            // Let this generation parent the next tool spans for the runner.
            $this->traceManager->scheduleToolParent($runnerId, $generation);
          }
        }
      }

      if (is_array($chatMessage)) {
        $outputData['output'] = ['raw' => $chatMessage];
      }
      elseif (is_object($chatMessage) && method_exists($chatMessage, 'getText') && $chatMessage->getText() !== '') {
        $outputData['output'] = $chatMessage->getText();
        $trace->update(output: $chatMessage->getText());
      }
      elseif (is_scalar($chatMessage) || (is_object($chatMessage) && method_exists($chatMessage, '__toString'))) {
        $outputData['output'] = (string) $chatMessage;
        $trace->update(output: (string) $chatMessage);
      }
      else {
        $outputData['output'] = NULL;
      }

      if (!isset($outputData['output']) && !empty($normalizedTools)) {
        $outputData['output'] = ['tool_calls' => $normalizedTools];
      }
    }
    catch (\Exception $e) {
      // Fallback to raw output if normalization fails.
      $this->logger->warning('Failed to get normalized output, using raw: @error', [
        '@error' => $e->getMessage(),
      ]);
      $outputData['output'] = $output->getRawOutput();
    }

    // Extract usage from raw for LangFuse analytics.
    $raw = $output->getRawOutput();
    if (is_array($raw) && isset($raw['usage'])) {
      // Use LangFuse standard 'usage_details' field for token analytics.
      if ($event->getOperationType() === 'embeddings') {
        // For embeddings, we want different usage format.
        $outputData['usage_details']['input'] = $raw['usage']['prompt_tokens'] ?? 0;
      }
      else {
        $outputData['usage_details'] = $raw['usage'];
      }
    }

    if (!array_key_exists('output', $outputData)) {
      $outputData['output'] = NULL;
    }

    // Add debug data if available.
    if ($event->getDebugData()) {
      $outputData['debug_details'] = $event->getDebugData();
    }

    // Add metadata if available and keep array structure for SDK validation.
    $responseMetadata = $event->getAllMetadata() ?: [];
    if (!empty($extraMetadata)) {
      $responseMetadata = array_merge($responseMetadata, $extraMetadata);
    }
    if (!empty($responseMetadata)) {
      $outputData['metadata'] = $responseMetadata;
    }

    $generation->end($outputData);
    unset($this->activeGenerations[$targetThreadId]);
    return TRUE;
  }

  /**
   * Determines whether the event output represents a streaming iterator.
   *
   * @param \Drupal\ai\Event\AiProviderResponseBaseEvent $event
   *   The AI provider response event.
   *
   * @return bool
   *   TRUE if the output is a streaming iterator, FALSE otherwise.
   */
  private function isStreamingOutput(AiProviderResponseBaseEvent $event): bool {
    $output = $event->getOutput();

    if (!method_exists($output, 'getNormalized')) {
      return FALSE;
    }

    try {
      $normalized = $output->getNormalized();
    }
    catch (\Throwable $throwable) {
      $this->logger->warning('Unable to inspect normalized output for streaming detection: @error', [
        '@error' => $throwable->getMessage(),
      ]);
      return FALSE;
    }

    return $normalized instanceof StreamedChatMessageIteratorInterface;
  }

  /**
   * Finalizes a trace with metadata and ends it.
   *
   * @param mixed $trace
   *   The trace object.
   * @param mixed $event
   *   The event object (PostGenerateResponseEvent or
   *   PostStreamingResponseEvent).
   * @param string $type
   *   The type of finalization ('response' or 'streaming').
   */
  private function finalizeTrace($trace, $event, string $type): void {
    $metadata = ["{$type}_timestamp" => time()];

    if ($type === 'response') {
      $metadata['response_type'] = $event->getOutput() ? get_class($event->getOutput()) : NULL;
      $metadata['debug_data_json'] = json_encode($event->getDebugData());
      $metadata['event_metadata_json'] = json_encode($event->getAllMetadata());
      $trace->end();
    }
    elseif ($type === 'streaming') {
      $metadata['streaming_complete'] = TRUE;
      $metadata['final_output_preview'] = $this->traceManager->createSafePreview($event->getOutput());
      $metadata['final_metadata_json'] = json_encode($event->getAllMetadata());
    }

    $trace->updateMetadata($metadata);
  }

  /**
   * Extracts an AI agent runner identifier from event tags.
   */
  private function extractRunnerIdFromTags(array $tags): ?string {
    $prefix = 'ai_agents_runner_';
    foreach ($tags as $key => $value) {
      if (is_string($value) && str_starts_with($value, $prefix)) {
        return substr($value, strlen($prefix));
      }
      if (is_string($key) && str_starts_with($key, $prefix) && $value) {
        return substr($key, strlen($prefix));
      }
    }
    return NULL;
  }

  /**
   * Converts tool call outputs to a structured array for logging.
   *
   * @param \Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutputInterface[] $tools
   *   Tool outputs returned by the AI response.
   *
   * @return array
   *   Normalized tool call data.
   */
  private function normalizeToolCalls(array $tools): array {
    $normalized = [];
    foreach ($tools as $tool) {
      if (!$tool instanceof ToolsFunctionOutputInterface) {
        continue;
      }

      $arguments = [];
      foreach ($tool->getArguments() as $argument) {
        if ($argument instanceof ToolsPropertyResultInterface) {
          $arguments[$argument->getName()] = $argument->getValue();
        }
      }

      $normalized[] = array_filter([
        'tool_id' => $tool->getToolId(),
        'name' => $tool->getName(),
        'arguments' => $arguments,
      ], static fn($value) => $value !== NULL && $value !== '' && $value !== []);
    }

    return $normalized;
  }

}
