<?php

namespace Drupal\langfuse_ai_search_logging\EventSubscriber;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\langfuse\LangFuseClientInterface;
use Drupal\langfuse\Service\LangFuseTraceManagerInterface;
use Drupal\search_api\Event\ProcessingResultsEvent;
use Drupal\search_api\Event\QueryPreExecuteEvent;
use Drupal\search_api\Event\SearchApiEvents;
use Dropsolid\LangFuse\Observability\ObservationInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribes to Search API events to capture LangFuse spans.
 */
final class LangFuseSearchSpanSubscriber implements EventSubscriberInterface {

  private const CONTEXT_RETRIEVAL = 'ai_search.retrieval';

  /**
   * Active search spans keyed by query object id.
   *
   * @var array<int,array{span:\Dropsolid\LangFuse\Observability\ObservationInterface, started:float}>
   */
  private array $activeSearchSpans = [];

  /**
   * Constructs the subscriber.
   */
  public function __construct(
    protected LangFuseClientInterface $langFuseClient,
    protected LangFuseTraceManagerInterface $traceManager,
    protected TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      SearchApiEvents::QUERY_PRE_EXECUTE => ['onSearchQueryPreExecute', 0],
      SearchApiEvents::PROCESSING_RESULTS => ['onSearchProcessingResults', 0],
    ];
  }

  /**
   * Starts a span for the AI Search query.
   */
  public function onSearchQueryPreExecute(QueryPreExecuteEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }
    $query = $event->getQuery();
    if (!$this->traceManager->isAiSearchQuery($query)) {
      return;
    }

    $index = $query->getIndex();
    ['trace' => $trace] = $this->traceManager->ensureTrace([
      'first_ai_provider' => 'ai_search_query',
      'first_operation_type' => 'retrieval',
      'first_search_index' => $index->id(),
    ], ['rag', 'search']);
    if (!$trace) {
      return;
    }

    $spanMetadata = $this->traceManager->buildSearchSpanMetadata($query);
    $spanInput = $this->traceManager->normalizeSearchQueryInput($query);
    if ($spanInput !== NULL) {
      $trace->update(input: $spanInput);
    }

    $span = $trace->createSpan('retrieval_ai_search', $spanMetadata, $spanInput);
    // Retrieval spans sit underneath the tool span context if it exists.
    $parentSpan = $this->traceManager->currentSpanContext('ai_search.tool');
    if ($parentSpan instanceof ObservationInterface) {
      $span->setParentObservationId($parentSpan->getId());
    }
    // Store retrieval span so embeddings can attach to it later.
    $this->traceManager->enterSpanContext(self::CONTEXT_RETRIEVAL, $span);
    $this->activeSearchSpans[$this->traceManager->getSearchSpanKey($query)] = [
      'span' => $span,
      'started' => microtime(TRUE),
    ];
  }

  /**
   * Completes the span once Search API returns results.
   */
  public function onSearchProcessingResults(ProcessingResultsEvent $event): void {
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    $results = $event->getResults();
    $query = $results->getQuery();
    if (!$this->traceManager->isAiSearchQuery($query)) {
      return;
    }

    $key = $this->traceManager->getSearchSpanKey($query);
    if (empty($this->activeSearchSpans[$key])) {
      return;
    }
    $span = $this->activeSearchSpans[$key]['span'];
    $started = $this->activeSearchSpans[$key]['started'];
    unset($this->activeSearchSpans[$key]);

    $metadata = array_filter([
      'status' => 'completed',
      'finished_timestamp' => $this->time->getRequestTime(),
      'duration_ms' => round(max(0, (microtime(TRUE) - $started) * 1000), 2),
      'result_count' => $results->getResultCount(),
      'warnings_preview' => $this->traceManager->createSafePreview($results->getWarnings()),
      'ignored_keys_preview' => $this->traceManager->createSafePreview($results->getIgnoredSearchKeys()),
    ], static fn($value) => $value !== NULL && $value !== '');

    $span->end([
      'metadata' => $metadata,
      'output' => [
        'result_ids' => array_slice(array_keys($results->getResultItems()), 0, 25),
        'query_options' => $query->getOptions(),
      ],
    ]);
    // Retrieval scope complete; embeddings can no longer attach to it.
    $this->traceManager->leaveSpanContext(self::CONTEXT_RETRIEVAL, $span);
  }

}
