<?php

namespace Drupal\langfuse\EventSubscriber;

use Drupal\langfuse\LangFuseClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpKernel\Event\TerminateEvent;

/**
 * Event Subscriber for syncing LangFuse traces at the end of the request.
 */
class LangfuseSyncSubscriber implements EventSubscriberInterface {

  /**
   * The LangFuse client service.
   *
   * @var \Drupal\langfuse\LangFuseClientInterface
   */
  protected LangFuseClientInterface $langFuseClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs a new LangfuseSyncSubscriber.
   *
   * @param \Drupal\langfuse\LangFuseClientInterface $langFuseClient
   *   The LangFuse client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   */
  public function __construct(LangFuseClientInterface $langFuseClient, LoggerChannelFactoryInterface $loggerFactory) {
    $this->langFuseClient = $langFuseClient;
    $this->logger = $loggerFactory->get('langfuse');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      // Kernel Terminate has lower priority to ensure other
      // operations complete.
      KernelEvents::TERMINATE => ['onKernelTerminate', 100],
    ];
  }

  /**
   * Handles the Kernel Terminate event.
   *
   * Finalizes and syncs the current trace at the end of each request.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The terminate event.
   */
  public function onKernelTerminate(TerminateEvent $event) {
    // Only attempt to sync traces if LangFuse is properly configured.
    if (!$this->langFuseClient->isConfigured()) {
      return;
    }

    if (!$this->finalizeActiveTrace()) {
      // No trace to finalize.
      return;
    }

    $this->langFuseClient->syncTraces();
  }

  /**
   * Finalizes the current LangFuse trace for this request lifecycle.
   *
   * @return bool
   *   TRUE if a trace was finalized and syncing should continue, FALSE if
   *   there was no trace or an error occurred during finalization.
   */
  protected function finalizeActiveTrace(): bool {
    $trace = $this->langFuseClient->getCurrentTrace();
    if (!$trace) {
      return FALSE;
    }

    try {
      $observations = $trace->getObservations();
      $lastOutput = NULL;

      // Find the last generation output for trace-level output.
      foreach (array_reverse($observations) as $observation) {
        if (!method_exists($observation, 'getType') || strtolower($observation->getType()) !== 'generation') {
          continue;
        }
        $obsOutput = method_exists($observation, 'getOutput') ? $observation->getOutput() : NULL;
        if ($obsOutput !== NULL) {
          $lastOutput = $obsOutput;
          break;
        }
      }

      $trace->updateMetadata([
        'drupal_request_complete' => TRUE,
        'final_timestamp' => time(),
        'total_ai_operations' => count($observations),
        'output' => $lastOutput ? json_encode($lastOutput) : NULL,
      ]);

      $trace->end();
    }
    catch (\Throwable $throwable) {
      $this->logger->error('Failed to finalize LangFuse trace: @message', [
        '@message' => $throwable->getMessage(),
      ]);
      return FALSE;
    }
    finally {
      $this->langFuseClient->clearCurrentTrace();
    }

    return TRUE;
  }

}
