<?php

namespace Drupal\langfuse\Service;

use Drupal\ai\Service\FunctionCalling\ExecutableFunctionCallInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\langfuse\LangFuseClientInterface;
use Dropsolid\LangFuse\Observability\ObservationInterface;
use Dropsolid\LangFuse\Observability\Trace;
use Drupal\search_api\Query\QueryInterface;

/**
 * Provides shared LangFuse trace utilities and runner context tracking.
 */
class LangFuseTraceManager implements LangFuseTraceManagerInterface {

  /**
   * Runner context metadata keyed by runner id.
   *
   * @var array<string,array{
   *   parent_runner:?string,
   *   pending_parent:?array{
   *     observation:\Dropsolid\LangFuse\Observability\ObservationInterface,
   *     usage:int
   *   },
   *   delegation_span:?\Dropsolid\LangFuse\Observability\ObservationInterface,
   * }>
   */
  protected array $runnerContexts = [];

  /**
   * Generic span stacks keyed by context name.
   *
   * @var array<string,\Dropsolid\LangFuse\Observability\ObservationInterface[]>
   */
  protected array $spanStacks = [];


  /**
   * The logger channel.
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructs the trace manager.
   */
  public function __construct(
    protected LangFuseClientInterface $langFuseClient,
    protected AccountProxyInterface $currentUser,
    LoggerChannelFactoryInterface $loggerFactory,
    protected TimeInterface $time,
  ) {
    $this->logger = $loggerFactory->get('langfuse_ai_logging');
  }

  /**
   * {@inheritdoc}
   */
  public function ensureTrace(array $metadata = [], array $tags = [], ?string $initialInput = NULL): array {
    if (!$this->langFuseClient->isConfigured()) {
      return ['trace' => NULL, 'created' => FALSE];
    }

    $trace = $this->langFuseClient->getCurrentTrace();
    $filteredMetadata = $this->filterMetadata($metadata);

    if ($trace instanceof Trace) {
      if ($filteredMetadata) {
        $existingMetadata = $trace->getMetadata() ?? [];
        $missingMetadata = [];
        foreach ($filteredMetadata as $key => $value) {
          if (!array_key_exists($key, $existingMetadata)) {
            $missingMetadata[$key] = $value;
          }
        }
        if ($missingMetadata) {
          $trace->updateMetadata($missingMetadata);
        }
      }
      return ['trace' => $trace, 'created' => FALSE];
    }

    $traceMetadata = array_merge([
      'drupal_request' => TRUE,
      'request_timestamp' => $this->time->getRequestTime(),
    ], $filteredMetadata);
    $traceTags = array_values(array_unique(array_merge(['ai', 'drupal', 'multi_operation'], $tags)));

    $trace = $this->langFuseClient->createTrace(
      'drupal_ai_request',
      (string) $this->currentUser->id(),
      session_id(),
      $traceMetadata,
      $traceTags,
      NULL,
      NULL,
    );

    if ($initialInput !== NULL && $initialInput !== '') {
      $trace->update(input: $initialInput);
    }

    $this->logger->info('Created new LangFuse trace @trace_id (@name) for Drupal AI request', [
      '@trace_id' => $trace->getId(),
      '@name' => 'drupal_ai_request',
    ]);

    return ['trace' => $trace, 'created' => TRUE];
  }

  /**
   * {@inheritdoc}
   */
  public function registerRunnerParent(string $runnerId, ?string $parentRunnerId): void {
    $context = &$this->getRunnerContext($runnerId);
    $context['parent_runner'] = $parentRunnerId;
  }

  /**
   * {@inheritdoc}
   */
  public function getRunnerParent(string $runnerId): ?string {
    return $this->runnerContexts[$runnerId]['parent_runner'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function clearRunnerContext(string $runnerId): void {
    unset($this->runnerContexts[$runnerId]);
  }

  /**
   * {@inheritdoc}
   */
  public function scheduleToolParent(string $runnerId, ObservationInterface $observation): void {
    $context = &$this->getRunnerContext($runnerId);
    $context['pending_parent'] = [
      'observation' => $observation,
      'usage' => 0,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function claimToolParent(string $runnerId): ?ObservationInterface {
    if (empty($this->runnerContexts[$runnerId]['pending_parent'])) {
      return NULL;
    }
    $this->runnerContexts[$runnerId]['pending_parent']['usage']++;
    return $this->runnerContexts[$runnerId]['pending_parent']['observation'];
  }

  /**
   * {@inheritdoc}
   */
  public function releaseToolParent(string $runnerId): void {
    if (empty($this->runnerContexts[$runnerId]['pending_parent'])) {
      return;
    }
    $this->runnerContexts[$runnerId]['pending_parent']['usage']--;
    if ($this->runnerContexts[$runnerId]['pending_parent']['usage'] <= 0) {
      $this->runnerContexts[$runnerId]['pending_parent'] = NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function startDelegation(string $runnerId, ObservationInterface $span): void {
    $context = &$this->getRunnerContext($runnerId);
    $context['delegation_span'] = $span;
  }

  /**
   * {@inheritdoc}
   */
  public function endDelegation(string $runnerId): void {
    if (isset($this->runnerContexts[$runnerId])) {
      $this->runnerContexts[$runnerId]['delegation_span'] = NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDelegationSpan(string $runnerId): ?ObservationInterface {
    return $this->runnerContexts[$runnerId]['delegation_span'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function enterSpanContext(string $context, ObservationInterface $span): void {
    $this->spanStacks[$context][] = $span;
  }

  /**
   * {@inheritdoc}
   */
  public function leaveSpanContext(string $context, ObservationInterface $span): void {
    if (empty($this->spanStacks[$context])) {
      return;
    }
    $current = array_pop($this->spanStacks[$context]);
    if ($current !== $span) {
      $this->spanStacks[$context] = array_values(
        array_filter($this->spanStacks[$context], static fn($item) => $item !== $span)
      );
    }
    if (empty($this->spanStacks[$context])) {
      unset($this->spanStacks[$context]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function currentSpanContext(string $context): ?ObservationInterface {
    if (empty($this->spanStacks[$context])) {
      return NULL;
    }
    return $this->spanStacks[$context][array_key_last($this->spanStacks[$context])];
  }

  /**
   * {@inheritdoc}
   */
  public function tagsContainValue(array $tags, string $needle): bool {
    if (isset($tags[$needle])) {
      return (bool) $tags[$needle];
    }
    return in_array($needle, $tags, TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function normalizeToolContextValues(ExecutableFunctionCallInterface $tool): array {
    $contextValues = [];
    foreach ($tool->getContextValues() as $key => $value) {
      $contextValues[$key] = $this->normalizeValue($value);
    }
    return $contextValues;
  }

  /**
   * {@inheritdoc}
   */
  public function buildSearchSpanMetadata(QueryInterface $query): array {
    return array_filter([
      'index_id' => $query->getIndex()->id(),
      'server_id' => $query->getIndex()->getServerId(),
      'limit' => $query->getOption('limit'),
      'offset' => $query->getOption('offset'),
      'tags_preview' => $this->createSafePreview($query->getTags()),
      'options_preview' => $this->createSafePreview($query->getOptions()),
      'chunk_result' => $query->getOption('search_api_ai_get_chunks_result'),
    ], static fn($value) => $value !== NULL && $value !== '');
  }

  /**
   * {@inheritdoc}
   */
  public function normalizeSearchQueryInput(QueryInterface $query): ?string {
    $keys = $query->getKeys();
    if (is_array($keys)) {
      unset($keys['#conjunction']);
      $keys = trim(implode(' ', array_map('strval', $keys)));
    }
    elseif (!is_string($keys)) {
      return NULL;
    }

    if ($keys === '') {
      return NULL;
    }

    return $keys;
  }

  /**
   * {@inheritdoc}
   */
  public function isAiSearchQuery(QueryInterface $query): bool {
    try {
      $server = $query->getIndex()->getServerInstance();
    }
    catch (\Exception $exception) {
      $this->logger->debug('Unable to load server for index @index: @error', [
        '@index' => $query->getIndex()->id(),
        '@error' => $exception->getMessage(),
      ]);
      return FALSE;
    }

    return $server->getBackendId() === 'search_api_ai_search';
  }

  /**
   * {@inheritdoc}
   */
  public function getSearchSpanKey(QueryInterface $query): string {
    return sprintf('search_query:%d', spl_object_id($query));
  }

  /**
   * {@inheritdoc}
   */
  public function createSafePreview(mixed $data, int $limit = 400): ?string {
    if ($data === NULL) {
      return NULL;
    }

    if (is_scalar($data)) {
      $string = (string) $data;
    }
    elseif (is_object($data) && method_exists($data, '__toString')) {
      $string = (string) $data;
    }
    else {
      $string = json_encode($data);
    }

    if ($string === FALSE) {
      return NULL;
    }

    if (mb_strlen($string) <= $limit) {
      return $string;
    }

    return mb_substr($string, 0, $limit) . '...';
  }

  /**
   * Helper to filter metadata arrays.
   */
  protected function filterMetadata(array $metadata): array {
    return array_filter($metadata, static fn($value) => $value !== NULL && $value !== '');
  }

  /**
   * Normalizes arbitrary context values.
   */
  protected function normalizeValue(mixed $value): mixed {
    if ($value === NULL || is_scalar($value)) {
      return $value;
    }

    if ($value instanceof \DateTimeInterface) {
      return $value->format(DATE_ATOM);
    }

    if (is_array($value)) {
      $normalized = [];
      foreach ($value as $key => $item) {
        $normalized[$key] = $this->normalizeValue($item);
      }
      return $normalized;
    }

    if ($value instanceof \JsonSerializable) {
      return $value->jsonSerialize();
    }

    if (is_object($value) && method_exists($value, '__toString')) {
      return (string) $value;
    }

    if (is_object($value)) {
      return sprintf('[object:%s]', get_class($value));
    }

    return (string) $value;
  }

  /**
   * Returns the mutable context array for a runner.
   */
  protected function &getRunnerContext(string $runnerId): array {
    if (!isset($this->runnerContexts[$runnerId])) {
      $this->runnerContexts[$runnerId] = [
        'parent_runner' => NULL,
        'pending_parent' => NULL,
        'delegation_span' => NULL,
      ];
    }
    return $this->runnerContexts[$runnerId];
  }
}
