<?php

namespace Drupal\language_negotiation_matrix\Plugin\LanguageNegotiation;

use Drupal\Core\Render\BubbleableMetadata;
use Drupal\language_negotiation_matrix\MatrixManager;
use Drupal\language\Plugin\LanguageNegotiation\LanguageNegotiationUrl;
use Symfony\Component\HttpFoundation\Request;

/**
 * Class for identifying language via URL prefix or domain.
 *
 * @LanguageNegotiation(
 *   id = \Drupal\language_negotiation_matrix\Plugin\LanguageNegotiation\LanguageNegotiationUrlMatrix::METHOD_ID,
 *   types = {\Drupal\Core\Language\LanguageInterface::TYPE_INTERFACE,
 *   \Drupal\Core\Language\LanguageInterface::TYPE_CONTENT,
 *   \Drupal\Core\Language\LanguageInterface::TYPE_URL},
 *   weight = -99,
 *   name = @Translation("URL Matrix"),
 *   description = @Translation("Language from the URL Matrix (Path prefix and/or domain)."),
 *   config_route_name = "language_negotiation_matrix.negotiation_url_matrix"
 * )
 */
class LanguageNegotiationUrlMatrix extends LanguageNegotiationUrl {

  /**
   * The language negotiation method id.
   */
  const METHOD_ID = 'language-url-matrix';

  /**
   * URL language negotiation: use the path prefix as URL language indicator.
   */
  const CONFIG_PATH_PREFIX = 'path_prefix';

  /**
   * URL language negotiation: use the domain as URL language indicator.
   */
  const CONFIG_DOMAIN = 'domain_matrix';

  /**
   * The language matrix manager.
   *
   * @var \Drupal\language_negotiation_matrix\MatrixManager
   */
  protected MatrixManager $matrixManager;

  /**
   * {@inheritdoc}
   */
  // @TODO: There's an odd issue that stems from flipping the base_url
  //        on an SSO plugin login process. Consider KeyCloak and
  //        OpenID Connect as an example: when trying to flip languages
  //        after login (en to fr), the cookie / session seems to be lost and we
  //        don't get a login created in the language we requested.
  //        Could have something to do with the CORS standard OR header
  //        language swapping. Can't be certain. Need to investigate.
  public function processOutbound($path, &$options = [], Request $request = NULL, BubbleableMetadata $bubbleable_metadata = NULL) {
    $path = parent::processOutbound($path, $options, $request, $bubbleable_metadata);
    $base_root = $request?->getSchemeAndHttpHost();
    $this->matrixManager = \Drupal::service('language_negotiation_matrix.matrix_manager');
    $langcode = $options['language']->getId();

    // Update the PATH PREFIX mapping
    $config = $this->config->get('language.negotiation')->get('matrix');
    if ($config['source'] == LanguageNegotiationUrlMatrix::CONFIG_PATH_PREFIX) {
      if (is_object($options['language']) && !empty($config['prefixes'][$options['language']->getId()])) {
        $mapping = $this->matrixManager->getSiteAliasMapping($config['prefixes']);
        if (!empty($mapping) && $this->matrixManager->siteAliasExists($mapping, $langcode)) {
          $options['base_url'] = '/' . trim($mapping[$langcode], '/');
        }
      }
    }
    // Update the DOMAIN PREFIX mapping
    elseif ($config['source'] == LanguageNegotiationUrlMatrix::CONFIG_DOMAIN) {
      if (is_object($options['language']) && !empty($config['domains'][$options['language']->getId()])) {
        $mapping = $this->matrixManager->getSiteAliasMapping($config['domains']);
        if ($mapping) {
          $options['base_url'] = $base_root . '/' . trim($mapping[$langcode], '/');
        }
        $url_components = parse_url($options['base_url']);
        if (isset($url_components['host'])) {
          $options['base_url'] = str_replace($url_components['host'], $config['domains'][$langcode], $options['base_url']);
        }
      }
    }
    return $path;
  }

}
