<?php

namespace Drupal\laposta_subscribe\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\laposta_subscribe\Service\LapostaSubscribeLogger;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\Exception\RequestException;

class LapostaSubscribeController extends ControllerBase
{

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The Laposta subscribe logger.
   *
   * @var \Drupal\laposta_subscribe\LapostaSubscribeLogger
   */
  protected $lapostaLogger;

  /**
   * The date formatter.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * LapostaSubscribeController constructor.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\laposta_subscribe\LapostaSubscribeLogger $laposta_logger
   *   The Laposta subscribe logger.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ClientInterface $http_client, LapostaSubscribeLogger $laposta_logger, DateFormatterInterface $date_formatter, ConfigFactoryInterface $config_factory)
  {
    $this->httpClient = $http_client;
    $this->lapostaLogger = $laposta_logger;
    $this->dateFormatter = $date_formatter;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('http_client'),
      $container->get('laposta_subscribe.logger'),
      $container->get('date.formatter'),
      $container->get('config.factory')
    );
  }

  public function subscribe(Request $request)
  {
    try {
      // Make use of Honeypot if module is installed
      $module_handler = \Drupal::service('module_handler');
      if ($module_handler->moduleExists('honeypot')) {
        // Get Honeypot settings
        $honeypot_config = \Drupal::config('honeypot.settings');
        $honeypot_time_limit = $honeypot_config->get('time_limit');
        $honeypot_element_name = $honeypot_config->get('element_name');

        $time_limit = 0;
        $current_time = time();

        // Get field values from form
        $honeypot_element = $request->request->get($honeypot_element_name);
        $honeypot_time = $request->request->get('honeypot_time');

        if (!empty($honeypot_element) || ($current_time - $honeypot_time < $honeypot_time_limit)) {
          return new JsonResponse(['error' => $this->t('Potential spam detected.')], 400);
        }
      }

      $email = $request->request->get('email');

      if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $this->getLogger('laposta_subscribe')->error('Invalid or empty email address submitted');
        return new JsonResponse([
          'error' => t('Please provide a valid email address.')->render(),
        ], 400);
      }

      // Get the API key, list ID, and enabled fields from the configuration
      $config = $this->config('laposta_subscribe.settings');
      $api_key = $config->get('api_key');
      $list_id = $config->get('list_id');
      $enabled_fields = $config->get('enabled_fields') ?: [];
      $available_fields = $config->get('available_fields') ?: [];

      // $this->getLogger('laposta_subscribe')->notice('API Key: @key, List ID: @list', ['@key' => $api_key, '@list' => $list_id]);
      // $this->getLogger('laposta_subscribe')->notice('Enabled fields: @fields', ['@fields' => json_encode($enabled_fields)]);

      if (empty($api_key) || empty($list_id)) {
        $this->getLogger('laposta_subscribe')->error('Configuration error: API key or List ID is missing');
        return new JsonResponse(['error' => $this->t('Configuration error. Please contact the site administrator.')], 500);
      }

      // Get the IP address
      $ip_address = $request->getClientIp();

      // Prepare the data for Laposta API
      $data = [
        'list_id' => $list_id,
        'email' => $email,
        'ip' => $ip_address,
      ];

      $field_order = $config->get('field_order') ?: [];

      // Handle custom fields
      $custom_fields = new \stdClass();
      foreach ($field_order as $field_tag) {
        if (in_array($field_tag, $enabled_fields) && isset($available_fields[$field_tag]) && $field_tag !== 'email') {
          $field_info = $available_fields[$field_tag];

          // Handle multiple values for select_multiple
          if ($field_info['data_type'] === 'select_multiple') {
            $field_value = $request->request->all()[$field_tag] ?? [];
          } else {
            $field_value = $request->request->get($field_tag);
          }

          // Process the field value based on its data type
          switch ($field_info['data_type']) {
            case 'numeric':
              $field_value = is_numeric($field_value) ? floatval($field_value) : null;
              break;
            case 'date':
              // Assuming the date is in YYYY-MM-DD format
              $field_value = date('Y-m-d', strtotime($field_value)) ?: null;
              break;
            case 'select_multiple':
              // Ensure it's an array and filter out empty values
              $field_value = is_array($field_value) ? array_values(array_filter($field_value)) : [$field_value];
              break;
            case 'select_single':
              // Ensure the value is a single string, not an array
              $field_value = is_array($field_value) ? reset($field_value) : $field_value;
              break;
            case 'text':
            default:
              // For text, we can use the value as-is
              $field_value = $field_value ?: null;
              break;
          }

          if ($field_value !== null && $field_value !== '' && $field_value !== []) {
            $custom_fields->{$field_tag} = $field_value;
          }
        }
      }

      // Only add custom_fields to data if there are any
      if (!empty((array) $custom_fields)) {
        $data['custom_fields'] = $custom_fields;
      }

      // $this->getLogger('laposta_subscribe')->notice('Field order: @order', ['@order' => json_encode($field_order)]);
      // $this->getLogger('laposta_subscribe')->notice('Custom fields: @fields', ['@fields' => json_encode($custom_fields)]);
      $this->getLogger('laposta_subscribe')->notice('Final data being sent to Laposta: @data', ['@data' => json_encode($data)]);

      // Send request to Laposta API
      $response = $this->httpClient->request('POST', 'https://api.laposta.nl/v2/member', [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
          'Content-Type' => 'application/json',
        ],
        'json' => $data,
      ]);

      $http_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);
      $status = 'subscribed'; // Default status

      $this->getLogger('laposta_subscribe')->notice('Laposta API Response: @response', ['@response' => json_encode($body)]);

      // Log the subscription using our custom logger
      $this->lapostaLogger->logSubscription($email, $ip_address, $status);

      // Always return a success message, regardless of the actual result
      return new JsonResponse([
        'message' => $this->t('Thank you for your subscription.'),
      ], 200);

    } catch (RequestException $e) {
      // Handle RequestException (which includes connection errors)
      $this->handleApiException($e, $email ?? '', $ip_address ?? '');
    } catch (\Exception $e) {
      // Handle any other exceptions
      $this->handleGeneralException($e, $email ?? '', $ip_address ?? '');
    }

    // Always return a success message, regardless of the actual result
    return new JsonResponse([
      'message' => $this->t('Thank you for your subscription.'),
    ], 200);
  }

  private function handleApiException(RequestException $e, $email, $ip_address)
  {
    if ($e->hasResponse()) {
      $statusCode = $e->getResponse()->getStatusCode();
      $body = json_decode($e->getResponse()->getBody()->getContents(), TRUE);

      $this->getLogger('laposta_subscribe')->error('Laposta API Response: @response', ['@response' => json_encode($body)]);

      if (isset($body['error']['code']) && $body['error']['code'] == 204) {
        $status = 'exists';
      } else {
        $status = 'api_error';
      }
    } else {
      $this->getLogger('laposta_subscribe')->error('Laposta API Connection Error: @error', ['@error' => $e->getMessage()]);
      $status = 'connection_error';
    }

    $this->lapostaLogger->logSubscription($email, $ip_address, $status);
  }

  private function handleGeneralException(\Exception $e, $email, $ip_address)
  {
    $this->getLogger('laposta_subscribe')->error('Laposta General Exception: @error', ['@error' => $e->getMessage()]);
    $status = 'unknown_error';
    $this->lapostaLogger->logSubscription($email, $ip_address, $status);
  }
}