<?php

namespace Drupal\laposta_subscribe\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;

/**
 * Defines the Laposta Subscription entity.
 *
 * @ContentEntityType(
 *   id = "laposta_subscription",
 *   label = @Translation("Laposta Subscription"),
 *   base_table = "laposta_subscriptions",
 *   entity_keys = {
 *     "id" = "id",
 *     "uuid" = "uuid"
 *   }
 * )
 */
class LapostaSubscription extends ContentEntityBase
{

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type)
  {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['email'] = BaseFieldDefinition::create('email')
      ->setLabel(t('Email'))
      ->setRequired(TRUE);

    $fields['ip_address'] = BaseFieldDefinition::create('string')
      ->setLabel(t('IP Address'))
      ->setRequired(TRUE);

    $fields['status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Status'))
      ->setDescription(t('The status of the subscription.'))
      ->setRequired(TRUE)
      ->setSetting('allowed_values', [
        'subscribed' => 'Subscribed',
        'exists' => 'Email already exists',
        'unknown error' => 'Unknown error',
      ])
      ->setDefaultValue('subscribed');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the subscription was created.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(EntityTypeInterface $entity_type)
  {
    return [
      'tables' => [
        'laposta_subscriptions' => [
          'fields' => [
            'id' => [
              'type' => 'serial',
              'not null' => TRUE,
              'description' => 'The primary identifier for a Laposta Subscription.',
            ],
            'uuid' => [
              'type' => 'varchar',
              'length' => 128,
              'not null' => TRUE,
              'description' => 'The UUID of the Laposta Subscription.',
            ],
            'email' => [
              'type' => 'varchar',
              'length' => 255,
              'not null' => TRUE,
              'description' => 'The email address of the subscriber.',
            ],
            'ip_address' => [
              'type' => 'varchar',
              'length' => 45,
              'not null' => TRUE,
              'description' => 'The IP address of the subscriber.',
            ],
            'status' => [
              'type' => 'varchar',
              'length' => 20,
              'not null' => TRUE,
              'default' => 'active',
              'description' => 'The status of the subscription.',
            ],
            'created' => [
              'type' => 'int',
              'not null' => TRUE,
              'description' => 'The timestamp when the subscription was created.',
            ],
          ],
          'primary key' => ['id'],
          'unique keys' => [
            'uuid' => ['uuid'],
          ],
          'indexes' => [
            'created' => ['created'],
          ],
        ],
      ],
    ];
  }

  public function getStatusLabel() {
    $status = $this->get('status')->value;
    $allowed_values = $this->getFieldDefinition('status')->getSetting('allowed_values');
    
    if (isset($allowed_values[$status])) {
      return $this->t($allowed_values[$status]);
    }
    
    return $status;
  }

  /**
   * Gets the creation timestamp of the subscription.
   *
   * @return int
   *   Creation timestamp of the subscription.
   */
  public function getCreatedTime()
  {
    return $this->get('created')->value;
  }

  /**
   * Sets the creation timestamp of the subscription.
   *
   * @param int $timestamp
   *   The subscription creation timestamp.
   *
   * @return \Drupal\laposta_subscribe\Entity\LapostaSubscription
   *   The called subscription entity.
   */
  public function setCreatedTime($timestamp)
  {
    $this->set('created', $timestamp);
    return $this;
  }
}