<?php

namespace Drupal\laposta_subscribe\Form;

use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class LapostaSubscribeApiKeyForm extends ConfigFormBase
{
  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a LapostaSubscribeApiKeyForm object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(ClientInterface $http_client)
  {
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['laposta_subscribe.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'laposta_api_key_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $config = $this->config('laposta_subscribe.settings');
    $api_key = $config->get('api_key');

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $api_key ? $this->maskApiKey($api_key) : '',
      '#description' => $this->t('<a href="@link" target="_blank">How do I get an API key?</a>', ['@link' => 'https://docs.laposta.nl/article/349-hoe-kom-ik-aan-een-api-sleutel']),
      '#attributes' => ['autocomplete' => 'off'],
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $submitted_api_key = $form_state->getValue('api_key');
    $config = $this->config('laposta_subscribe.settings');

    if (!$this->isApiKeyMasked($submitted_api_key)) {
      if ($this->validateApiKey($submitted_api_key)) {
        $config->set('api_key', $submitted_api_key)->save();
        $this->messenger()->addStatus($this->t('API key has been validated and saved.'));
        // Clear the default status message set by ConfigFormBase
        $this->messenger()->deleteByType('status');
      } else {
        $form_state->setErrorByName('api_key', $this->t('The provided API key is invalid. Please check and try again.'));
      }
    }

    parent::submitForm($form, $form_state);
  }

  private function validateApiKey($api_key)
  {
    try {
      $response = $this->httpClient->request('GET', 'https://api.laposta.nl/v2/list', [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        ],
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      return $status_code === 200;
    } catch (\Exception $e) {
      $this->logger->error('API key validation error: @error', ['@error' => $e->getMessage()]);
      return FALSE;
    }
  }

  private function maskApiKey($key)
  {
    $visible_chars = 4;
    $masked_length = max(strlen($key) - $visible_chars, 0);
    return str_repeat('•', $masked_length) . substr($key, -$visible_chars);
  }

  private function isApiKeyMasked($key)
  {
    return strpos($key, '•') !== false;
  }

  public function validateForm(array &$form, FormStateInterface $form_state)
  {
    parent::validateForm($form, $form_state);

    $submitted_api_key = $form_state->getValue('api_key');
    if (!$this->isApiKeyMasked($submitted_api_key)) {
      if (!$this->validateApiKey($submitted_api_key)) {
        $form_state->setErrorByName('api_key', $this->t('The provided API key is invalid. Please check and try again.'));
      }
    }
  }
}