<?php

namespace Drupal\laposta_subscribe\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class LapostaSubscribeForm extends FormBase
{
  public function getFormId()
  {
    return 'laposta_subscribe_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $form['#attached']['library'][] = 'laposta_subscribe/laposta_subscribe';

    $config = $this->config('laposta_subscribe.settings');
    $api_key = $config->get('api_key');
    $available_fields = $config->get('available_fields');
    $enabled_fields = $config->get('enabled_fields');
    $field_order = $config->get('field_order');
    $list_id = $config->get('list_id');

    if (empty($api_key)) {
      $form['message'] = [
        '#markup' => $this->t('Please set your API key first.'),
      ];
      return $form;
    }

    if (empty($list_id) || empty($enabled_fields)) {
      $form['message'] = [
        '#markup' => $this->t('Please select a list and enable some fields.'),
      ];
      return $form;
    }

    // Use list_name_override if available, otherwise fall back to list_name
    $list_name = $config->get('list_name_override') ?: $config->get('list_name');
    if (!empty($list_name)) {
      $form['list_name'] = [
        '#markup' => '<h2>' . $list_name . '</h2>',
      ];
    }

    // Show list remarks if enabled
    if ($config->get('show_list_remarks')) {
      $list_remarks = $config->get('list_remarks');
      if (!empty($list_remarks)) {
        $form['list_remarks'] = [
          '#markup' => '<div class="list-remarks">' . $list_remarks . '</div>',
        ];
      }
    }

    foreach ($field_order as $field_name) {
      if (in_array($field_name, $enabled_fields) && isset($available_fields[$field_name])) {
        $field = $available_fields[$field_name];
        $form_field = [
          '#type' => $this->mapFieldType($field['data_type']),
          '#title' => $field['name'],
          '#required' => !empty($field['required']),
        ];

        switch ($field['data_type']) {
          case 'select_single':
            if (!empty($field['options'])) {
              if (isset($field['datatype_display']) && $field['datatype_display'] === 'radio') {
                $form_field['#type'] = 'radios';
              } else {
                $form_field['#type'] = 'select';
              }
              $form_field['#options'] = array_combine($field['options'], $field['options']);
            }
            break;
          case 'select_multiple':
            if (!empty($field['options'])) {
              $form_field['#type'] = 'checkboxes';
              $form_field['#options'] = array_combine($field['options'], $field['options']);
            }
            break;
          case 'checkbox':
            $form_field['#type'] = 'checkbox';
            break;
          case 'date':
            $form_field['#type'] = 'date';
            break;
          case 'numeric':
            $form_field['#type'] = 'number';
            break;
        }

        $form[$field_name] = $form_field;
      }
    }

    // Use submit_button_text if available, otherwise use default text
    $submit_text = $config->get('submit_button_text') ?: $this->t('Subscribe');

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $submit_text,
    ];

    // Add privacy policy text
    $privacy_policy = $config->get('privacy_policy_text');
    if (!empty($privacy_policy['value'])) {
      $form['privacy_policy'] = [
        '#type' => 'processed_text',
        '#text' => $privacy_policy['value'],
        '#format' => $privacy_policy['format'],
        '#weight' => 100, // Ensure it appears after the submit button
      ];
    }

    return $form;
  }

  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $this->messenger()->addStatus($this->t('Form submitted successfully.'));
  }

  private function mapFieldType($lapostaType)
  {
    $map = [
      'text' => 'textfield',
      'numeric' => 'number',
      'date' => 'date',
      'select_single' => 'select',
      'select_multiple' => 'checkboxes',
      'checkbox' => 'checkbox',
    ];

    return $map[$lapostaType] ?? 'textfield';
  }
}