<?php

namespace Drupal\laposta_subscribe\Form;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Cache\Cache;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class LapostaSubscribeListFieldsForm extends ConfigFormBase
{

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;
  protected $logger;
  protected $dateFormatter;

  public function __construct(
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory,
    DateFormatterInterface $date_formatter
  ) {
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('laposta_subscribe');
    $this->dateFormatter = $date_formatter;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('http_client'),
      $container->get('logger.factory'),
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames()
  {
    return ['laposta_subscribe.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId()
  {
    return 'laposta_subscribe_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state)
  {
    $form['#attached']['library'][] = 'laposta_subscribe/admin_styles';

    $config = $this->config('laposta_subscribe.settings');
    $api_key = $config->get('api_key');

    if (empty($api_key)) {
      $form['message'] = [
        '#markup' => $this->t('Please set your API key first.'),
      ];
      return $form;
    }

    $available_lists = $this->fetchListsFromApi($api_key);

    if (empty($available_lists)) {
      $form['message'] = [
        '#markup' => $this->t('No lists found. Please create a list in your Laposta account.'),
      ];
      return $form;
    }

    $form['list_select'] = [
      '#type' => 'select',
      '#title' => $this->t('Select List'),
      '#options' => $available_lists,
      '#empty_option' => $this->t('- Select a list -'),
      '#default_value' => $form_state->get('list_id') ?: $config->get('list_id'),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::updateListDetails',
        'wrapper' => 'list-details-wrapper',
      ],
    ];

    $form['list_details_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'list-details-wrapper'],
    ];

    $selected_list_id = $form_state->get('list_id') ?: $config->get('list_id');
    if ($selected_list_id && isset($available_lists[$selected_list_id])) {
      $list_details = $this->fetchListDetailsFromApi($api_key, $selected_list_id);
      $form['list_details_wrapper'] += $this->buildListDetailsForm($list_details);
    } else {
      $form['list_details_wrapper']['message'] = [
        '#markup' => $this->t('Please select a list to view details.'),
      ];
    }

    $this->initializeFieldOrder();

    // Add field descriptions
    $form['field_descriptions'] = [
      '#type' => 'details',
      '#title' => $this->t('Field Descriptions'),
      '#open' => FALSE,
      '#weight' => 10,
    ];

    $readable_names = $this->getReadableFieldNames();
    foreach ($this->getFieldDescriptions() as $field => $description) {
      $form['field_descriptions'][$field] = [
        '#type' => 'item',
        '#title' => isset($readable_names[$field]) ? $readable_names[$field] : $this->t('@field', ['@field' => ucfirst($field)]),
        '#description' => $description,
      ];
    }

    $form['fields_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'fields-wrapper'],
    ];

    $form['fields_wrapper']['field_table'] = $this->buildFieldsTable();

    $form['fetch_fields'] = [
      '#type' => 'submit',
      '#value' => $this->t('Fetch Available Fields'),
      '#submit' => ['::fetchFields'],
      '#ajax' => [
        'callback' => '::updateFieldsTable',
        'wrapper' => 'fields-wrapper',
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to update list details.
   */
  public function updateListDetails(array &$form, FormStateInterface $form_state)
  {
    $list_id = $form_state->getValue('list_select');
    $config = $this->config('laposta_subscribe.settings');
    $api_key = $config->get('api_key');

    if (empty($list_id)) {
      $element = [
        '#markup' => $this->t('No list selected.'),
      ];
    } else {
      $list_details = $this->fetchListDetailsFromApi($api_key, $list_id);
      $element = [
        '#type' => 'container',
        '#attributes' => ['id' => 'list-details-wrapper'],
        'details' => $this->buildListDetailsForm($list_details),
      ];
    }

    $form_state->set('list_id', $list_id);
    $config->set('list_id', $list_id)->save();

    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#list-details-wrapper', $element));

    return $response;
  }

  /**
   * Builds the list details form elements.
   */
  protected function buildListDetailsForm($list_details)
  {
    $element = [
      '#cache' => ['max-age' => 0],  // Disable caching for this element
    ];

    if (!empty($list_details['created'])) {
      $created_date = new \DateTime($list_details['created']);
      $formatted_date = $this->dateFormatter->format($created_date->getTimestamp(), 'medium');
      $element['created'] = [
        '#type' => 'item',
        '#title' => $this->t('Created'),
        '#markup' => $formatted_date,
      ];
    }

    if (!empty($list_details['name'])) {
      $element['list_name'] = [
        '#type' => 'item',
        '#title' => $this->t('List Name'),
        '#markup' => $list_details['name'],
      ];

      $element['list_remarks'] = [
        '#type' => 'item',
        '#title' => $this->t('List Remarks'),
        '#markup' => !empty($list_details['remarks']) ? $list_details['remarks'] : $this->t('No remarks available.'),
      ];

      if (!empty($list_details['members'])) {
        $element['members_stats'] = [
          '#type' => 'table',
          '#header' => [
            $this->t('Members'),
            $this->t('Active'),
            $this->t('Unsubscribed'),
            $this->t('Cleaned'),
          ],
          '#rows' => [
            [
              $this->t('Count'),
              $list_details['members']['active'] ?? 0,
              $list_details['members']['unsubscribed'] ?? 0,
              $list_details['members']['cleaned'] ?? 0,
            ],
          ],
          '#attributes' => ['class' => ['member-statistics-table']],
        ];
      }
    } else {
      $element['no_list'] = [
        '#markup' => $this->t('Failed to fetch list details or no list selected.'),
      ];
    }

    return $element;
  }

  /**
   * Fetches lists from the Laposta API.
   */
  protected function fetchListsFromApi($api_key)
  {
    try {
      $response = $this->httpClient->request('GET', "https://api.laposta.nl/v2/list", [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        ],
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);

      if ($status_code === 200 && isset($body['data']) && is_array($body['data'])) {
        $lists = [];
        foreach ($body['data'] as $list_data) {
          if (
            isset($list_data['list']) &&
            isset($list_data['list']['list_id']) &&
            isset($list_data['list']['name']) &&
            isset($list_data['list']['state']) &&
            $list_data['list']['state'] === 'active'
          ) {
            $lists[$list_data['list']['list_id']] = $list_data['list']['name'];
          }
        }
        asort($lists);
        return $lists;
      } else {
        $error_message = isset($body['error']['message']) ? $body['error']['message'] : 'Unknown error';
        $error_code = isset($body['error']['code']) ? $body['error']['code'] : 'Unknown code';
        $this->messenger()->addWarning($this->t('Failed to fetch lists. Error: @error (Code: @code)', [
          '@error' => $error_message,
          '@code' => $error_code,
        ]));
        $this->logger->warning('Failed to fetch lists. Status code: @status, Error: @error (Code: @code)', [
          '@status' => $status_code,
          '@error' => $error_message,
          '@code' => $error_code,
        ]);
      }
    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while fetching lists.'));
      $this->logger->error('Error fetching lists: @error', ['@error' => $e->getMessage()]);
    }

    return [];
  }

  /**
   * Fetches list details from the Laposta API.
   */
  protected function fetchListDetailsFromApi($api_key, $list_id)
  {
    try {
      $response = $this->httpClient->request('GET', "https://api.laposta.nl/v2/list/{$list_id}", [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        ],
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);

      if ($status_code === 200 && isset($body['list'])) {
        $list_details = [
          'created' => $body['list']['created'] ?? '',
          'name' => $body['list']['name'] ?? '',
          'remarks' => $body['list']['remarks'] ?? '',
          'members' => $body['list']['members'] ?? [],
        ];
        return $list_details;
      } else {
        $error_message = isset($body['error']['message']) ? $body['error']['message'] : 'Unknown error';
        $error_code = isset($body['error']['code']) ? $body['error']['code'] : 'Unknown code';
        $this->messenger()->addWarning($this->t('Failed to fetch list details. Error: @error (Code: @code)', [
          '@error' => $error_message,
          '@code' => $error_code,
        ]));
        $this->logger->warning('Failed to fetch list details. Status code: @status, Error: @error (Code: @code)', [
          '@status' => $status_code,
          '@error' => $error_message,
          '@code' => $error_code,
        ]);
      }
    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while fetching list details.'));
      $this->logger->error('Error fetching list details: @error', ['@error' => $e->getMessage()]);
    }

    return ['name' => '', 'remarks' => ''];
  }

  /**
   * Builds the fields table.
   */
  protected function buildFieldsTable()
  {
    $config = $this->config('laposta_subscribe.settings');
    $available_fields = $config->get('available_fields') ?: [];
    $enabled_fields = $config->get('enabled_fields') ?: [];
    $field_order = $config->get('field_order') ?: array_keys($available_fields);

    $table = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field'),
        $this->t('Data Type'),
        $this->t('Required'),
        $this->t('Enabled'),
        $this->t('Weight'),
      ],
      '#empty' => $this->t('No fields available. Please fetch fields for the selected list.'),
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'field-weight',
        ],
      ],
    ];

    foreach ($field_order as $weight => $field_name) {
      if (isset($available_fields[$field_name])) {
        $field_info = $available_fields[$field_name];
        $is_required = $field_info['required'] ?? FALSE;

        $table[$field_name] = [
          '#attributes' => ['class' => ['draggable']],
          'name' => ['#markup' => $field_info['name']],
          'data_type' => [
            '#markup' => $this->getReadableFieldType($field_info['data_type']) .
              (isset($field_info['datatype_display']) && $field_info['datatype_display']
                ? ' (' . $this->t('radio') . ')'
                : ''
              ),
          ],
          'required' => ['#markup' => $is_required ? $this->t('Yes') : $this->t('No')],
          'enabled' => [
            '#type' => 'checkbox',
            '#default_value' => in_array($field_name, $enabled_fields) || $is_required,
            '#disabled' => $is_required,
          ],
          'weight' => [
            '#type' => 'weight',
            '#title' => $this->t('Weight for @title', ['@title' => $field_info['name']]),
            '#title_display' => 'invisible',
            '#default_value' => $weight,
            '#attributes' => ['class' => ['field-weight']],
          ],
        ];
      }
    }

    return $table;
  }

  /**
   * Ajax callback to update the fields table.
   */
  public function updateFieldsTable(array &$form, FormStateInterface $form_state)
  {
    return $form['fields_wrapper'];
  }

  /**
   * Get initial enabled fields including required fields and email.
   */
  protected function getInitialEnabledFields($fields)
  {
    $enabled_fields = [];
    foreach ($fields as $field_name => $field_info) {
      if ($field_info['required']) {
        $enabled_fields[] = $field_name;
      }
    }
    return $enabled_fields;
  }

  /**
   * Submit handler for fetching fields.
   */
  public function fetchFields(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('laposta_subscribe.settings');
    $api_key = $config->get('api_key');
    $list_id = $form_state->getValue('list_select');

    if (empty($list_id)) {
      $this->messenger()->addError($this->t('Please select a list before fetching fields.'));
      return;
    }

    $fields = $this->fetchFieldsFromApi($api_key, $list_id);

    if (!empty($fields)) {
      $config->set('available_fields', $fields);

      // Initialize the field order with the new fields
      $field_order = array_keys($fields);
      $config->set('field_order', $field_order);

      // Initialize enabled_fields with required fields and email
      $enabled_fields = $this->getInitialEnabledFields($fields);
      $config->set('enabled_fields', $enabled_fields);

      $config->save();

      $this->messenger()->addStatus($this->t('Fields fetched successfully.'));

      // Rebuild the entire form
      $form_state->setRebuild();
    } else {
      $this->messenger()->addWarning($this->t('No fields were found or there was an error fetching fields.'));
    }
  }

  /**
   * Fetches fields from the Laposta API.
   */
  protected function fetchFieldsFromApi($api_key, $list_id)
  {
    try {
      $response = $this->httpClient->request('GET', "https://api.laposta.nl/v2/field", [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        ],
        'query' => [
          'list_id' => $list_id,
        ],
        'http_errors' => FALSE,
      ]);

      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);

      if ($status_code === 200 && isset($body['data']) && is_array($body['data'])) {
        $fields = [];
        foreach ($body['data'] as $field_data) {
          if (isset($field_data['field'])) {
            $field = $field_data['field'];
            $field_info = [
              'name' => $field['name'],
              'required' => $field['required'] ?? FALSE,
              'is_email' => $field['is_email'] ?? FALSE,
              'data_type' => $field['datatype'] ?? 'text',
              'datatype_display' => $field['datatype_display'] ?? FALSE,
            ];

            // Add options for select_single and select_multiple fields
            if (in_array($field['datatype'], ['select_single', 'select_multiple']) && isset($field['options'])) {
              $field_info['options'] = $field['options'];
            }

            $fields[trim($field['tag'], '{}')] = $field_info;
          }
        }
        return $fields;
      } else {
        $this->messenger()->addWarning($this->t('Failed to fetch fields.'));
        $this->logger->warning('Failed to fetch fields. Status code: @status', ['@status' => $status_code]);
      }
    } catch (\Exception $e) {
      $this->messenger()->addError($this->t('An error occurred while fetching fields.'));
      $this->logger->error('Error fetching fields: @error', ['@error' => $e->getMessage()]);
    }

    return [];
  }

  /**
   * Gets a human-readable field type.
   */
  protected function getReadableFieldType($data_type)
  {
    $types = [
      'text' => $this->t('Text'),
      'numeric' => $this->t('Number'),
      'date' => $this->t('Date'),
      'select_single' => $this->t('Single Select'),
      'select_multiple' => $this->t('Multiple Select'),
      'checkbox' => $this->t('Checkbox'),
    ];

    return $types[$data_type] ?? $this->t('Unknown');
  }

  protected function initializeFieldOrder()
  {
    $config = $this->config('laposta_subscribe.settings');
    $available_fields = $config->get('available_fields') ?: [];
    $field_order = $config->get('field_order');

    if (empty($field_order)) {
      $field_order = array_keys($available_fields);
      $config->set('field_order', $field_order)->save();
    }

    return $field_order;
  }

  protected function getReadableFieldNames()
  {
    return [
      'field' => $this->t('Field'),
      'data_type' => $this->t('Data Type'),
      'required' => $this->t('Required'),
      'enabled' => $this->t('Enabled'),
      'weight' => $this->t('Weight'),
    ];
  }

  protected function getFieldDescriptions()
  {
    return [
      'field' => $this->t('The name of the field as it appears in Laposta.'),
      'data_type' => $this->t('The type of data this field accepts.'),
      'required' => $this->t('Indicates whether this field is required in the subscription form.'),
      'enabled' => $this->t('Check this box to include the field in your subscription form.'),
      'weight' => $this->t('Drag to reorder the fields. Fields with lower weights appear first in the form.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state)
  {
    $config = $this->config('laposta_subscribe.settings');
    $list_id = $form_state->getValue('list_select');
    $config->set('list_id', $list_id);

    $available_fields = $config->get('available_fields') ?: [];
    $enabled_fields = $this->getInitialEnabledFields($available_fields);
    $field_order = [];

    // Fetch list details
    $api_key = $config->get('api_key');
    $list_details = $this->fetchListDetailsFromApi($api_key, $list_id);

    // Explicitly set list_name in configuration
    if (!empty($list_details['name'])) {
      $config->set('list_name', $list_details['name']);
      // Debug log
      $this->logger->debug('Setting list_name in config: @name', ['@name' => $list_details['name']]);
    } else {
      $this->logger->warning('List name is empty for list ID: @id', ['@id' => $list_id]);
    }

    // Save other list details
    $config->set('list_remarks', $list_details['remarks'] ?? '');

    foreach ($form_state->getValue('field_table') as $field_name => $field_data) {
      if (isset($available_fields[$field_name])) {
        // If the field is enabled by the user or required, add it to enabled_fields if is not already in enabled_fields
        if (!in_array($field_name, $enabled_fields)) {
          if ($field_data['enabled'] || $available_fields[$field_name]['required']) {
            $enabled_fields[] = $field_name;
          }
        }
        $field_order[$field_data['weight']] = $field_name;

        $available_fields[$field_name]['datatype_display'] = $available_fields[$field_name]['datatype_display'] ?? FALSE;
      }
    }

    ksort($field_order);

    $config->set('enabled_fields', $enabled_fields);
    $config->set('field_order', array_values($field_order));

    $config->save();

    // Clear caches
    Cache::invalidateTags($config->getCacheTags());

    parent::submitForm($form, $form_state);
  }

}