<?php

namespace Drupal\laposta_subscribe\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

class LapostaSubscribeApiService
{
  protected $configFactory;
  protected $httpClient;
  protected $stringTranslation;
  protected $loggerFactory;

  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    TranslationInterface $string_translation,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->stringTranslation = $string_translation;
    $this->loggerFactory = $logger_factory;
  }

  public function getApiInfo()
  {
    $config = $this->configFactory->get('laposta_subscribe.settings');
    $api_key = $config->get('api_key');
    $api_url = 'https://api.laposta.nl/v2/auth';

    if (empty($api_key)) {
      return [
        'status' => 'error',
        'message' => $this->stringTranslation->translate('API key not set'),
        'account_number' => null,
        'account_name' => null,
      ];
    }

    try {
      $response = $this->httpClient->request('GET', $api_url, [
        'headers' => [
          'Authorization' => 'Basic ' . base64_encode($api_key . ':'),
        ],
      ]);

      $statusCode = $response->getStatusCode();
      if ($statusCode == 200) {
        $data = json_decode($response->getBody(), true);
        $authenticated = $data['authenticated'] ?? false;
        $account_number = $data['account_number'] ?? null;
        $account_name = $data['account_name'] ?? null;

        if ($authenticated) {
          return [
            'status' => 'success',
            'message' => $this->stringTranslation->translate('API connection successful'),
            'account_number' => $account_number,
            'account_name' => $account_name,
          ];
        } else {
          $this->loggerFactory->get('laposta_subscribe')->warning('Laposta API authentication failed');
          return [
            'status' => 'error',
            'message' => $this->stringTranslation->translate('Authentication failed'),
            'account_number' => null,
            'account_name' => null,
          ];
        }
      } else {
        $this->loggerFactory->get('laposta_subscribe')->error('Failed to connect to Laposta API: @error', ['@error' => $response->getReasonPhrase()]);
        return [
          'status' => 'error',
          'message' => $this->stringTranslation->translate('Failed to connect to API: @error', ['@error' => $response->getReasonPhrase()]),
          'account_number' => null,
          'account_name' => null,
        ];
      }
    } catch (\Exception $e) {
      $this->loggerFactory->get('laposta_subscribe')->error('Exception occurred while connecting to Laposta API: @error', ['@error' => $e->getMessage()]);
      return [
        'status' => 'error',
        'message' => $this->stringTranslation->translate('Failed to connect to API: @error', ['@error' => $e->getMessage()]),
        'account_number' => null,
        'account_name' => null,
      ];
    }
  }
}