<?php

namespace Drupal\laposta_subscribe\Service;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

class LapostaSubscribeLogger
{
  protected $entityTypeManager;
  protected $dateFormatter;

  public function __construct(EntityTypeManagerInterface $entity_type_manager, DateFormatterInterface $date_formatter)
  {
    $this->entityTypeManager = $entity_type_manager;
    $this->dateFormatter = $date_formatter;
  }

  public static function create(ContainerInterface $container)
  {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('date.formatter')
    );
  }

  public function logSubscription($email, $ip_address, $status = 'subscribed')
  {
    $validStatuses = ['exists', 'subscribed', 'error'];

    // Ensure the status is valid
    if (!in_array($status, $validStatuses)) {
      $status = 'error';
    }

    // Anonymize the IP address before storing
    $anonymized_ip = $this->anonymizeIp($ip_address);

    try {
      $storage = $this->entityTypeManager->getStorage('laposta_subscription');
      $subscription = $storage->create([
        'email' => $email,
        'ip_address' => $anonymized_ip, // Store the anonymized IP
        'status' => $status,
      ]);
      $subscription->save();
      return $subscription->id();
    } catch (\Exception $e) {
      \Drupal::logger('laposta_subscribe')->error('Failed to log subscription: @message', ['@message' => $e->getMessage()]);
      return FALSE;
    }
  }

  public function getRecentSubscriptions($limit = 50)
  {
    $result = []; // Initialize an empty array

    try {
      $storage = $this->entityTypeManager->getStorage('laposta_subscription');
      $query = $storage->getQuery()
        ->sort('created', 'DESC')
        ->range(0, $limit)
        ->accessCheck(FALSE);

      $ids = $query->execute();
      $subscriptions = $storage->loadMultiple($ids);

      foreach ($subscriptions as $subscription) {
        $formatted_date = $this->dateFormatter->format($subscription->getCreatedTime(), 'medium');
        $result[] = [
          'email' => $subscription->get('email')->value,
          'ip_address' => $subscription->get('ip_address')->value,
          'status' => $this->getStatusMessage($subscription->get('status')->value),
          'created' => $formatted_date,
        ];
      }
    } catch (\Exception $e) {
      \Drupal::logger('laposta_subscribe')->error('Error fetching subscriptions: @error', ['@error' => $e->getMessage()]);
    }

    return $result;
  }

  protected function getStatusMessage($status)
  {
    $statusMessages = [
      'exists' => 'The email already exists',
      'subscribed' => 'Subscribed',
      'error' => 'Unknown error',
    ];

    return $statusMessages[$status] ?? 'Unknown status';
  }

  protected function anonymizeIp($ip_address)
  {
    // Special cases
    $specialIps = [
      '127.0.0.1' => '127.0.0.1',
      '::1' => '::1',
    ];

    if (isset($specialIps[$ip_address])) {
      return $specialIps[$ip_address];
    }

    if (filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
      // Anonymize IPv4 address
      $parts = explode('.', $ip_address);
      $parts[2] = 'xxx';
      $parts[3] = 'xxx';
      return implode('.', $parts);
    } elseif (filter_var($ip_address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)) {
      // Anonymize IPv6 address
      if (strpos($ip_address, '::') !== false) {
        // Handle shortened IPv6 addresses
        $parts = explode(':', $ip_address);
        $parts = array_pad($parts, 8, '0000');
      } else {
        $parts = explode(':', $ip_address);
      }
      $parts[6] = 'xxxx';
      $parts[7] = 'xxxx';
      return implode(':', $parts);
    } else {
      // Return a placeholder if the IP is not valid
      return 'invalid.ip.address';
    }
  }

  public function cleanupOldLogs($max_items = 20)
  {
    try {
      $storage = $this->entityTypeManager->getStorage('laposta_subscription');
      $query = $storage->getQuery()
        ->sort('created', 'ASC') // Oldest first
        ->accessCheck(FALSE);

      $ids = $query->execute();
      if (count($ids) > $max_items) {
        $ids_to_delete = array_slice($ids, 0, count($ids) - $max_items);
        $storage->delete($storage->loadMultiple($ids_to_delete));
      }
    } catch (\Exception $e) {
      \Drupal::logger('laposta_subscribe')->error('Failed to clean up old logs: @message', ['@message' => $e->getMessage()]);
    }
  }
}