<?php

namespace Drupal\laposta_webform\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\laposta_webform\Service\LapostaApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Laposta Webform settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The Laposta API service.
   *
   * @var \Drupal\laposta_webform\Service\LapostaApi
   */
  protected $lapostaApi;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\laposta_webform\Service\LapostaApi $laposta_api
   *   The Laposta API service.
   */
  public function __construct(LapostaApi $laposta_api) {
    $this->lapostaApi = $laposta_api;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('laposta_webform.api')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'laposta_webform_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['laposta_webform.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('laposta_webform.settings');
    $api_key = $config->get('api_key');

    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#default_value' => $api_key ? $this->maskApiKey($api_key) : '',
      '#description' => $this->t('Enter your Laposta API key. <a href="@link" target="_blank">How do I get an API key?</a>', [
        '@link' => 'https://docs.laposta.nl/article/349-hoe-kom-ik-aan-een-api-sleutel',
      ]),
      '#attributes' => ['autocomplete' => 'off'],
      '#required' => TRUE,
    ];

    if ($api_key) {
      $lists = $this->lapostaApi->getLists();
      if (!empty($lists)) {
        $form['connection_status'] = [
          '#type' => 'item',
          '#title' => $this->t('Connection Status'),
          '#markup' => '<span style="color: green;">' . $this->t('Connected - @count list(s) available', ['@count' => count($lists)]) . '</span>',
        ];
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $submitted_api_key = $form_state->getValue('api_key');

    // Don't validate if the key is masked (unchanged).
    if ($this->isApiKeyMasked($submitted_api_key)) {
      return;
    }

    if (!$this->lapostaApi->validateApiKey($submitted_api_key)) {
      $form_state->setErrorByName('api_key', $this->t('The provided API key is invalid. Please check and try again.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $submitted_api_key = $form_state->getValue('api_key');

    // Only save if the key is not masked (i.e., it was changed).
    if (!$this->isApiKeyMasked($submitted_api_key)) {
      $this->config('laposta_webform.settings')
        ->set('api_key', $submitted_api_key)
        ->save();

      $this->messenger()->addStatus($this->t('API key has been validated and saved.'));
    }

    parent::submitForm($form, $form_state);
  }

  /**
   * Masks an API key for display.
   *
   * @param string $key
   *   The API key.
   *
   * @return string
   *   The masked API key.
   */
  protected function maskApiKey(string $key): string {
    $visible_chars = 4;
    $masked_length = max(strlen($key) - $visible_chars, 0);
    return str_repeat('*', $masked_length) . substr($key, -$visible_chars);
  }

  /**
   * Checks if an API key is masked.
   *
   * @param string $key
   *   The API key to check.
   *
   * @return bool
   *   TRUE if the key is masked, FALSE otherwise.
   */
  protected function isApiKeyMasked(string $key): bool {
    return strpos($key, '*') !== FALSE;
  }

}
