<?php

namespace Drupal\laposta_webform\Plugin\WebformHandler;

use Drupal\Core\Form\FormStateInterface;
use Drupal\laposta_webform\Service\LapostaApi;
use Drupal\webform\Plugin\WebformHandlerBase;
use Drupal\webform\WebformSubmissionInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Webform submission handler for Laposta mailing lists.
 *
 * @WebformHandler(
 *   id = "laposta",
 *   label = @Translation("Laposta"),
 *   category = @Translation("External"),
 *   description = @Translation("Sends webform submissions to Laposta mailing lists."),
 *   cardinality = \Drupal\webform\Plugin\WebformHandlerInterface::CARDINALITY_UNLIMITED,
 *   results = \Drupal\webform\Plugin\WebformHandlerInterface::RESULTS_PROCESSED,
 * )
 */
class LapostaWebformHandler extends WebformHandlerBase {

  /**
   * The Laposta API service.
   *
   * @var \Drupal\laposta_webform\Service\LapostaApi
   */
  protected $lapostaApi;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->lapostaApi = $container->get('laposta_webform.api');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'list_id' => '',
      'list_field' => '',
      'email_field' => '',
      'optin_field' => '',
      'field_mapping' => [],
      'required_fields' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $api_key = $this->lapostaApi->getApiKey();

    if (empty($api_key)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('Please configure your Laposta API key first at <a href=":url">Laposta Webform settings</a>.', [
          ':url' => '/admin/config/services/laposta-webform',
        ]) . '</p>',
      ];
      return $form;
    }

    // Get available lists from Laposta.
    $lists = $this->lapostaApi->getLists();
    if (empty($lists)) {
      $form['message'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('No lists found in your Laposta account. Please create a list first.') . '</p>',
      ];
      return $form;
    }

    // Get webform elements for field mapping.
    $webform = $this->getWebform();
    $elements = $webform->getElementsDecodedAndFlattened();
    $element_options = ['' => $this->t('- Select -')];
    $email_options = ['' => $this->t('- Select -')];
    $optin_options = ['' => $this->t('- None (always subscribe) -')];
    $select_options = ['' => $this->t('- None (use fixed list) -')];

    foreach ($elements as $key => $element) {
      $title = $element['#title'] ?? $key;
      $element_options[$key] = $title;

      // Email fields.
      if (isset($element['#type']) && $element['#type'] === 'email') {
        $email_options[$key] = $title;
      }

      // Checkbox fields for opt-in.
      if (isset($element['#type']) && in_array($element['#type'], ['checkbox', 'webform_terms_of_service'])) {
        $optin_options[$key] = $title;
      }

      // Select/checkboxes fields for list selection.
      if (isset($element['#type']) && in_array($element['#type'], ['select', 'radios', 'checkboxes', 'webform_select_other', 'laposta_list_select', 'laposta_list_checkboxes'])) {
        $select_options[$key] = $title;
      }
    }

    // List selection options.
    $form['list_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('List Settings'),
    ];

    $form['list_settings']['list_field'] = [
      '#type' => 'select',
      '#title' => $this->t('List Selection Field'),
      '#description' => $this->t('Select a webform field that allows users to choose which list(s) to subscribe to. Leave empty to use a fixed list.'),
      '#options' => $select_options,
      '#default_value' => $this->configuration['list_field'],
    ];

    $form['list_settings']['list_id'] = [
      '#type' => 'select',
      '#title' => $this->t('Fixed List'),
      '#description' => $this->t('Select the Laposta list to subscribe users to. This is used when no list selection field is configured.'),
      '#options' => ['' => $this->t('- Select -')] + $lists,
      '#default_value' => $this->configuration['list_id'],
      '#states' => [
        'visible' => [
          ':input[name="settings[list_settings][list_field]"]' => ['value' => ''],
        ],
      ],
    ];

    // Email field.
    $form['email_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Email Field'),
      '#description' => $this->t('Select the webform field that contains the email address.'),
      '#options' => $email_options,
      '#default_value' => $this->configuration['email_field'],
      '#required' => TRUE,
    ];

    // Opt-in field.
    $form['optin_field'] = [
      '#type' => 'select',
      '#title' => $this->t('Opt-in Field'),
      '#description' => $this->t('Select a checkbox field that users must check to be subscribed. Leave empty to always subscribe.'),
      '#options' => $optin_options,
      '#default_value' => $this->configuration['optin_field'],
    ];

    // Field mapping section - only show when using a fixed list.
    $form['field_mapping'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Field Mapping'),
      '#description' => $this->t('Map webform fields to Laposta custom fields. Fields marked with * are required.'),
      '#states' => [
        'visible' => [
          ':input[name="settings[list_settings][list_field]"]' => ['value' => ''],
        ],
      ],
    ];

    // Get Laposta fields from the selected list only.
    $selected_list = $this->configuration['list_id'];
    $laposta_fields = [];

    if (!empty($selected_list)) {
      $laposta_fields = $this->lapostaApi->getFields($selected_list);
    }

    if (!empty($laposta_fields)) {
      // Build options and track required fields.
      $laposta_field_options = ['' => $this->t('- Do not map -')];
      $required_fields = [];

      foreach ($laposta_fields as $tag => $field_info) {
        // Skip email field as it's mapped separately.
        if (!empty($field_info['is_email'])) {
          continue;
        }
        $type_label = $this->getFieldTypeLabel($field_info['datatype']);
        $required_marker = $field_info['required'] ? ' *' : '';
        $laposta_field_options[$tag] = $field_info['name'] . ' (' . $type_label . ')' . $required_marker;

        if ($field_info['required']) {
          $required_fields[$tag] = $field_info['name'];
        }
      }

      // Show required fields info.
      if (!empty($required_fields)) {
        $form['field_mapping']['required_info'] = [
          '#type' => 'item',
          '#title' => $this->t('Required Laposta fields'),
          '#markup' => '<strong>' . implode(', ', $required_fields) . '</strong>',
          '#description' => $this->t('These fields must be mapped for successful subscription.'),
        ];
      }

      // Auto-mapping: try to match webform fields to Laposta fields by name.
      $auto_mapping = $this->getAutoMapping($elements, $laposta_fields);

      foreach ($elements as $key => $element) {
        // Skip email fields and special elements.
        $type = $element['#type'] ?? '';
        if ($type === 'email' || in_array($type, ['webform_actions', 'webform_flexbox', 'container', 'fieldset', 'details'])) {
          continue;
        }

        $title = $element['#title'] ?? $key;
        $default_value = $this->configuration['field_mapping'][$key] ?? $auto_mapping[$key] ?? '';

        $form['field_mapping'][$key] = [
          '#type' => 'select',
          '#title' => $title,
          '#options' => $laposta_field_options,
          '#default_value' => $default_value,
        ];
      }

      // Check for unmapped required fields and show warning.
      $current_mapping = $this->configuration['field_mapping'];
      $mapped_tags = array_values(array_filter($current_mapping));
      $unmapped_required = array_diff_key($required_fields, array_flip($mapped_tags));

      if (!empty($unmapped_required) && !empty($current_mapping)) {
        $form['field_mapping']['warning'] = [
          '#type' => 'item',
          '#markup' => '<div class="messages messages--warning">' . $this->t('Warning: The following required fields are not mapped: @fields', [
            '@fields' => implode(', ', $unmapped_required),
          ]) . '</div>',
          '#weight' => -10,
        ];
      }
    }
    else {
      $form['field_mapping']['info'] = [
        '#markup' => '<p>' . $this->t('Please select a list to see available Laposta fields.') . '</p>',
      ];
    }

    // Info message when using list selector.
    $form['list_selector_info'] = [
      '#type' => 'container',
      '#states' => [
        'visible' => [
          ':input[name="settings[list_settings][list_field]"]' => ['!value' => ''],
        ],
      ],
    ];
    $form['list_selector_info']['message'] = [
      '#type' => 'item',
      '#title' => $this->t('Field Mapping'),
      '#markup' => '<p>' . $this->t('When using a list selector, fields are automatically mapped by matching webform field names to Laposta field names/tags (case-insensitive). Make sure your webform field names match the Laposta field names.') . '</p>',
    ];

    return $form;
  }

  /**
   * Gets auto-mapping suggestions based on field name matching.
   *
   * @param array $webform_elements
   *   The webform elements.
   * @param array $laposta_fields
   *   The Laposta fields.
   *
   * @return array
   *   An array of suggested mappings keyed by webform field key.
   */
  protected function getAutoMapping(array $webform_elements, array $laposta_fields): array {
    $mapping = [];

    // Build a lookup array for Laposta fields by lowercase name and tag.
    $laposta_lookup = [];
    foreach ($laposta_fields as $tag => $field_info) {
      if (!empty($field_info['is_email'])) {
        continue;
      }
      $laposta_lookup[strtolower($tag)] = $tag;
      $laposta_lookup[strtolower($field_info['name'])] = $tag;
    }

    foreach ($webform_elements as $key => $element) {
      $type = $element['#type'] ?? '';
      if ($type === 'email') {
        continue;
      }

      // Try to match by key or title.
      $title = $element['#title'] ?? '';
      $key_lower = strtolower($key);
      $title_lower = strtolower($title);

      if (isset($laposta_lookup[$key_lower])) {
        $mapping[$key] = $laposta_lookup[$key_lower];
      }
      elseif (isset($laposta_lookup[$title_lower])) {
        $mapping[$key] = $laposta_lookup[$title_lower];
      }
    }

    return $mapping;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $this->configuration['list_id'] = $form_state->getValue(['list_settings', 'list_id']);
    $this->configuration['list_field'] = $form_state->getValue(['list_settings', 'list_field']);
    $this->configuration['email_field'] = $form_state->getValue('email_field');
    $this->configuration['optin_field'] = $form_state->getValue('optin_field');

    // When using list selector, clear field mapping (auto-map at submission time).
    if (!empty($this->configuration['list_field'])) {
      $this->configuration['field_mapping'] = [];
      $this->configuration['required_fields'] = [];
    }
    else {
      // Save field mapping for fixed list.
      $field_mapping = $form_state->getValue('field_mapping');
      if (is_array($field_mapping)) {
        // Filter out empty mappings.
        $this->configuration['field_mapping'] = array_filter($field_mapping);
      }

      // Save required Laposta fields for validation.
      $this->configuration['required_fields'] = $this->getRequiredLapostaFields();

      // Update webform elements to mark mapped required fields as required.
      $this->updateWebformRequiredFields();
    }
  }

  /**
   * Updates webform elements to mark fields mapped to required Laposta fields.
   */
  protected function updateWebformRequiredFields(): void {
    $field_mapping = $this->configuration['field_mapping'];
    $required_fields = $this->configuration['required_fields'] ?? [];

    if (empty($required_fields) || empty($field_mapping)) {
      return;
    }

    // Build reverse mapping: Laposta tag => webform field.
    $reverse_mapping = array_flip($field_mapping);

    // Get webform fields that should be required.
    $fields_to_require = [];
    foreach ($required_fields as $tag => $name) {
      if (isset($reverse_mapping[$tag])) {
        $fields_to_require[] = $reverse_mapping[$tag];
      }
    }

    if (empty($fields_to_require)) {
      return;
    }

    // Get the webform and update elements.
    $webform = $this->getWebform();
    $elements = $webform->getElementsDecoded();
    $changed = FALSE;

    foreach ($fields_to_require as $field_key) {
      if ($this->setWebformElementRequired($elements, $field_key)) {
        $changed = TRUE;
      }
    }

    if ($changed) {
      $webform->setElements($elements);
      $webform->save();
    }
  }

  /**
   * Recursively finds and marks a webform element as required.
   *
   * @param array &$elements
   *   The webform elements array.
   * @param string $field_key
   *   The field key to find.
   *
   * @return bool
   *   TRUE if the element was found and updated, FALSE otherwise.
   */
  protected function setWebformElementRequired(array &$elements, string $field_key): bool {
    foreach ($elements as $key => &$element) {
      if (!is_array($element)) {
        continue;
      }

      if ($key === $field_key) {
        // Only update if not already required.
        if (empty($element['#required'])) {
          $element['#required'] = TRUE;
          return TRUE;
        }
        return FALSE;
      }

      // Recurse into child elements (skip properties starting with #).
      foreach ($element as $child_key => &$child_element) {
        if (!str_starts_with($child_key, '#') && is_array($child_element)) {
          if ($this->setWebformElementRequired($element, $field_key)) {
            return TRUE;
          }
          break;
        }
      }
    }

    return FALSE;
  }

  /**
   * Gets required Laposta fields based on current configuration.
   *
   * @return array
   *   An array of required field tags with their names.
   */
  protected function getRequiredLapostaFields(): array {
    $required_fields = [];
    $lists = $this->lapostaApi->getLists();
    $list_field = $this->configuration['list_field'];
    $selected_list = $this->configuration['list_id'];

    if (!empty($list_field)) {
      // Using list selector - get required fields from all lists.
      foreach ($lists as $lid => $lname) {
        $list_fields = $this->lapostaApi->getFields($lid);
        foreach ($list_fields as $tag => $field_info) {
          if (!empty($field_info['required']) && empty($field_info['is_email'])) {
            $required_fields[$tag] = $field_info['name'];
          }
        }
      }
    }
    elseif (!empty($selected_list)) {
      // Fixed list - get required fields from that list.
      $list_fields = $this->lapostaApi->getFields($selected_list);
      foreach ($list_fields as $tag => $field_info) {
        if (!empty($field_info['required']) && empty($field_info['is_email'])) {
          $required_fields[$tag] = $field_info['name'];
        }
      }
    }

    return $required_fields;
  }

  /**
   * {@inheritdoc}
   */
  public function alterForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    // Get the field mapping and required fields.
    $field_mapping = $this->configuration['field_mapping'];
    $required_fields = $this->configuration['required_fields'] ?? [];

    if (empty($required_fields)) {
      return;
    }

    // Build reverse mapping: Laposta tag => webform field.
    $reverse_mapping = array_flip($field_mapping);

    // Mark webform fields as required if they're mapped to required Laposta fields.
    foreach ($required_fields as $tag => $name) {
      if (!isset($reverse_mapping[$tag])) {
        continue;
      }

      $webform_field = $reverse_mapping[$tag];

      // Find and mark the element as required.
      if (isset($form['elements'][$webform_field])) {
        $form['elements'][$webform_field]['#required'] = TRUE;
      }
      // Also check nested elements (in containers/fieldsets).
      $this->setElementRequired($form['elements'], $webform_field);
    }
  }

  /**
   * Recursively finds and marks an element as required.
   *
   * @param array &$elements
   *   The form elements array.
   * @param string $field_key
   *   The field key to find.
   */
  protected function setElementRequired(array &$elements, string $field_key): void {
    foreach ($elements as $key => &$element) {
      if ($key === $field_key && is_array($element)) {
        $element['#required'] = TRUE;
        return;
      }
      // Recurse into child elements.
      if (is_array($element) && !str_starts_with($key, '#')) {
        $this->setElementRequired($element, $field_key);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state, WebformSubmissionInterface $webform_submission) {
    // Check if opt-in is required and not checked - skip validation if so.
    $optin_field = $this->configuration['optin_field'];
    if (!empty($optin_field)) {
      $optin_value = $form_state->getValue($optin_field);
      if (empty($optin_value)) {
        // User did not opt in, no need to validate Laposta fields.
        return;
      }
    }

    $list_field = $this->configuration['list_field'];

    if (!empty($list_field)) {
      // Using list selector - validate required fields for the selected list(s).
      $this->validateDynamicListFields($form_state);
    }
    else {
      // Using fixed list - validate configured required fields.
      $this->validateFixedListFields($form_state);
    }
  }

  /**
   * Validates required fields when using a dynamic list selector.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateDynamicListFields(FormStateInterface $form_state): void {
    $list_field = $this->configuration['list_field'];
    $selected_lists = $form_state->getValue($list_field);

    if (empty($selected_lists)) {
      return;
    }

    // Normalize to array and filter empty values (unchecked checkboxes).
    if (!is_array($selected_lists)) {
      $selected_lists = [$selected_lists];
    }
    else {
      $selected_lists = array_filter($selected_lists);
    }

    if (empty($selected_lists)) {
      return;
    }

    // Get webform elements for matching.
    $webform = $this->getWebform();
    $elements = $webform->getElementsDecodedAndFlattened();

    // Get list names for error messages.
    $lists = $this->lapostaApi->getLists();

    // Check required fields for each selected list.
    foreach ($selected_lists as $list_id) {
      if (empty($list_id)) {
        continue;
      }

      $laposta_fields = $this->lapostaApi->getFields($list_id);
      $list_name = $lists[$list_id] ?? $list_id;

      foreach ($laposta_fields as $tag => $field_info) {
        if (empty($field_info['required']) || !empty($field_info['is_email'])) {
          continue;
        }

        // Find matching webform field by name.
        $webform_field = $this->findMatchingWebformField($tag, $field_info['name'], $elements);

        if ($webform_field === NULL) {
          // Required field has no matching webform field - show error.
          $form_state->setErrorByName($list_field, $this->t('The list "@list" requires the field "@field" but no matching webform field was found. Please add a field named "@tag" to your form.', [
            '@list' => $list_name,
            '@field' => $field_info['name'],
            '@tag' => $tag,
          ]));
          continue;
        }

        $value = $form_state->getValue($webform_field);

        if ($this->isValueEmpty($value)) {
          $form_state->setErrorByName($webform_field, $this->t('The field @field is required for newsletter subscription to "@list".', [
            '@field' => $field_info['name'],
            '@list' => $list_name,
          ]));
        }
      }
    }
  }

  /**
   * Finds a matching webform field by Laposta tag or name.
   *
   * @param string $tag
   *   The Laposta field tag.
   * @param string $name
   *   The Laposta field name.
   * @param array $elements
   *   The webform elements.
   *
   * @return string|null
   *   The matching webform field key, or NULL if not found.
   */
  protected function findMatchingWebformField(string $tag, string $name, array $elements): ?string {
    $tag_lower = strtolower($tag);
    $name_lower = strtolower($name);

    foreach ($elements as $key => $element) {
      $key_lower = strtolower($key);
      $title_lower = strtolower($element['#title'] ?? '');

      if ($key_lower === $tag_lower || $key_lower === $name_lower ||
          $title_lower === $tag_lower || $title_lower === $name_lower) {
        return $key;
      }
    }

    return NULL;
  }

  /**
   * Validates required fields when using a fixed list.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateFixedListFields(FormStateInterface $form_state): void {
    $field_mapping = $this->configuration['field_mapping'];
    $required_fields = $this->configuration['required_fields'] ?? [];

    // Build reverse mapping: Laposta tag => webform field.
    $reverse_mapping = array_flip($field_mapping);

    // Check each required Laposta field.
    foreach ($required_fields as $tag => $name) {
      if (!isset($reverse_mapping[$tag])) {
        // Required field is not mapped - this is a configuration issue.
        continue;
      }

      $webform_field = $reverse_mapping[$tag];
      $value = $form_state->getValue($webform_field);

      // Check if the value is empty.
      if ($this->isValueEmpty($value)) {
        $form_state->setErrorByName($webform_field, $this->t('The field @field is required for newsletter subscription.', [
          '@field' => $name,
        ]));
      }
    }
  }

  /**
   * Checks if a value is considered empty.
   *
   * @param mixed $value
   *   The value to check.
   *
   * @return bool
   *   TRUE if the value is empty, FALSE otherwise.
   */
  protected function isValueEmpty($value): bool {
    if ($value === NULL || $value === '' || $value === []) {
      return TRUE;
    }
    if (is_array($value)) {
      return empty(array_filter($value));
    }
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(WebformSubmissionInterface $webform_submission, $update = TRUE) {
    // Only process new submissions.
    if ($update) {
      return;
    }

    $data = $webform_submission->getData();

    // Check opt-in field if configured.
    $optin_field = $this->configuration['optin_field'];
    if (!empty($optin_field)) {
      $optin_value = $data[$optin_field] ?? FALSE;
      if (empty($optin_value)) {
        // User did not opt in, skip subscription.
        return;
      }
    }

    // Get email address.
    $email_field = $this->configuration['email_field'];
    $email = $data[$email_field] ?? '';
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $this->getLogger('laposta_webform')->warning('Invalid or missing email address for webform submission @sid', [
        '@sid' => $webform_submission->id(),
      ]);
      return;
    }

    // Determine list(s) to subscribe to.
    $list_ids = $this->getListIds($data);
    if (empty($list_ids)) {
      $this->getLogger('laposta_webform')->warning('No list configured for Laposta handler on webform @webform', [
        '@webform' => $this->getWebform()->id(),
      ]);
      return;
    }

    // Get list names for logging.
    $lists = $this->lapostaApi->getLists();

    // Subscribe to each list.
    foreach ($list_ids as $list_id) {
      // Build custom fields (with auto-mapping for the specific list if using list selector).
      $custom_fields = $this->buildCustomFields($data, $list_id);

      $result = $this->lapostaApi->subscribe($list_id, $email, $custom_fields);
      $list_name = $lists[$list_id] ?? $list_id;

      if ($result['success']) {
        $this->getLogger('laposta_webform')->info('Successfully subscribed @email to list @list', [
          '@email' => $email,
          '@list' => $list_name,
        ]);
      }
      else {
        $this->getLogger('laposta_webform')->error('Failed to subscribe @email to list @list: @error', [
          '@email' => $email,
          '@list' => $list_name,
          '@error' => $result['error'] ?? 'Unknown error',
        ]);
      }
    }
  }

  /**
   * Gets the list IDs to subscribe to.
   *
   * @param array $data
   *   The webform submission data.
   *
   * @return array
   *   An array of list IDs.
   */
  protected function getListIds(array $data): array {
    $list_field = $this->configuration['list_field'];

    // If a list field is configured, use its value(s).
    if (!empty($list_field) && isset($data[$list_field])) {
      $value = $data[$list_field];
      if (is_array($value)) {
        return array_filter($value);
      }
      if (!empty($value)) {
        return [$value];
      }
    }

    // Fall back to the fixed list ID.
    $list_id = $this->configuration['list_id'];
    if (!empty($list_id)) {
      return [$list_id];
    }

    return [];
  }

  /**
   * Builds the custom fields array for the API.
   *
   * @param array $data
   *   The webform submission data.
   * @param string $list_id
   *   The Laposta list ID to subscribe to.
   *
   * @return array
   *   An array of custom field values keyed by Laposta field tag.
   */
  protected function buildCustomFields(array $data, string $list_id): array {
    $custom_fields = [];

    // Determine the mapping to use.
    $list_field = $this->configuration['list_field'];

    if (!empty($list_field)) {
      // Using list selector - auto-map by field names for the specific list.
      $mapping = $this->getAutoMappingForList($list_id, $data);
    }
    else {
      // Using fixed list - use configured mapping.
      $mapping = $this->configuration['field_mapping'];
    }

    foreach ($mapping as $webform_field => $laposta_tag) {
      if (empty($laposta_tag) || !isset($data[$webform_field])) {
        continue;
      }

      $value = $data[$webform_field];

      // Handle different value types.
      if (is_array($value)) {
        // Multiple select values - keep as array for Laposta (select_multiple).
        $value = array_values(array_filter($value));
        if (!empty($value)) {
          $custom_fields[$laposta_tag] = $value;
        }
      }
      elseif ($value !== '' && $value !== NULL) {
        $custom_fields[$laposta_tag] = $value;
      }
    }

    return $custom_fields;
  }

  /**
   * Gets auto-mapping for a specific list based on submission data.
   *
   * @param string $list_id
   *   The Laposta list ID.
   * @param array $data
   *   The webform submission data.
   *
   * @return array
   *   An array of mappings keyed by webform field key with Laposta tag as value.
   */
  protected function getAutoMappingForList(string $list_id, array $data): array {
    $mapping = [];
    $laposta_fields = $this->lapostaApi->getFields($list_id);

    if (empty($laposta_fields)) {
      return $mapping;
    }

    // Build a lookup array for Laposta fields by lowercase name and tag.
    $laposta_lookup = [];
    foreach ($laposta_fields as $tag => $field_info) {
      if (!empty($field_info['is_email'])) {
        continue;
      }
      $laposta_lookup[strtolower($tag)] = $tag;
      $laposta_lookup[strtolower($field_info['name'])] = $tag;
    }

    // Get webform elements.
    $webform = $this->getWebform();
    $elements = $webform->getElementsDecodedAndFlattened();

    // Match webform fields to Laposta fields by name.
    foreach ($data as $field_key => $value) {
      // Skip fields that have no data or are special fields.
      if ($value === '' || $value === NULL || (is_array($value) && empty($value))) {
        continue;
      }

      // Get the element info.
      $element = $elements[$field_key] ?? NULL;
      if (!$element) {
        continue;
      }

      // Try to match by key or title.
      $title = $element['#title'] ?? '';
      $key_lower = strtolower($field_key);
      $title_lower = strtolower($title);

      if (isset($laposta_lookup[$key_lower])) {
        $mapping[$field_key] = $laposta_lookup[$key_lower];
      }
      elseif (isset($laposta_lookup[$title_lower])) {
        $mapping[$field_key] = $laposta_lookup[$title_lower];
      }
    }

    return $mapping;
  }

  /**
   * Gets a human-readable field type label.
   *
   * @param string $datatype
   *   The Laposta field datatype.
   *
   * @return string
   *   The human-readable label.
   */
  protected function getFieldTypeLabel(string $datatype): string {
    $types = [
      'text' => $this->t('Text'),
      'numeric' => $this->t('Number'),
      'date' => $this->t('Date'),
      'select_single' => $this->t('Single Select'),
      'select_multiple' => $this->t('Multiple Select'),
    ];

    return $types[$datatype] ?? $datatype;
  }

  /**
   * {@inheritdoc}
   */
  public function getSummary() {
    $settings = [];

    if (!empty($this->configuration['list_id'])) {
      $lists = $this->lapostaApi->getLists();
      $list_name = $lists[$this->configuration['list_id']] ?? $this->configuration['list_id'];
      $settings[] = $this->t('List: @list', ['@list' => $list_name]);
    }

    if (!empty($this->configuration['list_field'])) {
      $settings[] = $this->t('List field: @field', ['@field' => $this->configuration['list_field']]);
    }

    if (!empty($this->configuration['email_field'])) {
      $settings[] = $this->t('Email field: @field', ['@field' => $this->configuration['email_field']]);
    }

    if (!empty($this->configuration['optin_field'])) {
      $settings[] = $this->t('Opt-in field: @field', ['@field' => $this->configuration['optin_field']]);
    }

    if (!empty($this->configuration['list_field'])) {
      $settings[] = $this->t('Field mapping: Auto-map by name');
    }
    else {
      $mapping_count = count(array_filter($this->configuration['field_mapping']));
      if ($mapping_count > 0) {
        $settings[] = $this->t('@count field(s) mapped', ['@count' => $mapping_count]);
      }
    }

    return [
      '#markup' => implode('<br>', $settings),
    ];
  }

}
