<?php

namespace Drupal\laposta_webform\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;

/**
 * Service for interacting with the Laposta API.
 */
class LapostaApi {

  /**
   * The Laposta API base URL.
   */
  protected const API_BASE_URL = 'https://api.laposta.nl/v2';

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * Constructs a LapostaApi object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->logger = $logger_factory->get('laposta_webform');
  }

  /**
   * Gets the API key from configuration.
   *
   * @return string|null
   *   The API key or NULL if not set.
   */
  public function getApiKey(): ?string {
    return $this->configFactory->get('laposta_webform.settings')->get('api_key');
  }

  /**
   * Gets the authorization header value.
   *
   * @param string|null $api_key
   *   Optional API key. If not provided, uses the stored key.
   *
   * @return string
   *   The Authorization header value.
   */
  protected function getAuthHeader(?string $api_key = NULL): string {
    $key = $api_key ?? $this->getApiKey();
    return 'Basic ' . base64_encode($key . ':');
  }

  /**
   * Makes a request to the Laposta API.
   *
   * @param string $method
   *   The HTTP method.
   * @param string $endpoint
   *   The API endpoint (without base URL).
   * @param array $options
   *   Additional Guzzle options.
   * @param string|null $api_key
   *   Optional API key for validation purposes.
   *
   * @return array|null
   *   The decoded JSON response or NULL on error.
   */
  protected function request(string $method, string $endpoint, array $options = [], ?string $api_key = NULL): ?array {
    $url = self::API_BASE_URL . $endpoint;
    $options['headers']['Authorization'] = $this->getAuthHeader($api_key);
    $options['http_errors'] = FALSE;

    try {
      $response = $this->httpClient->request($method, $url, $options);
      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);

      if ($status_code >= 200 && $status_code < 300) {
        return $body;
      }

      $error_message = $body['error']['message'] ?? 'Unknown error';
      $error_code = $body['error']['code'] ?? $status_code;
      $this->logger->warning('Laposta API error: @error (Code: @code)', [
        '@error' => $error_message,
        '@code' => $error_code,
      ]);

      return NULL;
    }
    catch (RequestException $e) {
      $this->logger->error('Laposta API request failed: @error', [
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Validates an API key.
   *
   * @param string $api_key
   *   The API key to validate.
   *
   * @return bool
   *   TRUE if the API key is valid, FALSE otherwise.
   */
  public function validateApiKey(string $api_key): bool {
    $result = $this->request('GET', '/list', [], $api_key);
    return $result !== NULL;
  }

  /**
   * Gets all available lists.
   *
   * @return array
   *   An array of lists keyed by list_id with name as value.
   */
  public function getLists(): array {
    $result = $this->request('GET', '/list');

    if ($result === NULL || !isset($result['data'])) {
      return [];
    }

    $lists = [];
    foreach ($result['data'] as $list_data) {
      if (isset($list_data['list']) &&
          isset($list_data['list']['list_id']) &&
          isset($list_data['list']['name']) &&
          ($list_data['list']['state'] ?? '') === 'active') {
        $lists[$list_data['list']['list_id']] = $list_data['list']['name'];
      }
    }

    asort($lists);
    return $lists;
  }

  /**
   * Gets detailed information about a specific list.
   *
   * @param string $list_id
   *   The list ID.
   *
   * @return array|null
   *   The list details or NULL on error.
   */
  public function getListDetails(string $list_id): ?array {
    $result = $this->request('GET', '/list/' . $list_id);

    if ($result === NULL || !isset($result['list'])) {
      return NULL;
    }

    return [
      'list_id' => $result['list']['list_id'] ?? $list_id,
      'name' => $result['list']['name'] ?? '',
      'remarks' => $result['list']['remarks'] ?? '',
      'created' => $result['list']['created'] ?? '',
      'members' => $result['list']['members'] ?? [],
    ];
  }

  /**
   * Gets all fields for a specific list.
   *
   * @param string $list_id
   *   The list ID.
   *
   * @return array
   *   An array of fields with field tag as key.
   */
  public function getFields(string $list_id): array {
    $result = $this->request('GET', '/field', [
      'query' => ['list_id' => $list_id],
    ]);

    if ($result === NULL || !isset($result['data'])) {
      return [];
    }

    $fields = [];
    foreach ($result['data'] as $field_data) {
      if (isset($field_data['field'])) {
        $field = $field_data['field'];
        $tag = trim($field['tag'] ?? '', '{}');

        $fields[$tag] = [
          'name' => $field['name'] ?? $tag,
          'tag' => $tag,
          'required' => $field['required'] ?? FALSE,
          'is_email' => $field['is_email'] ?? FALSE,
          'datatype' => $field['datatype'] ?? 'text',
          'datatype_display' => $field['datatype_display'] ?? '',
          'options' => $field['options'] ?? [],
        ];
      }
    }

    return $fields;
  }

  /**
   * Subscribes a member to a list.
   *
   * @param string $list_id
   *   The list ID.
   * @param string $email
   *   The email address.
   * @param array $custom_fields
   *   An array of custom field values keyed by field tag.
   * @param string|null $ip
   *   The subscriber's IP address.
   *
   * @return array
   *   An array with 'success' boolean and 'message' or 'error' string.
   */
  public function subscribe(string $list_id, string $email, array $custom_fields = [], ?string $ip = NULL): array {
    $data = [
      'list_id' => $list_id,
      'email' => $email,
      'ip' => $ip ?? \Drupal::request()->getClientIp(),
    ];

    // Process custom fields for API submission using stdClass (same as Laposta Subscribe).
    if (!empty($custom_fields)) {
      $processed_fields = new \stdClass();
      foreach ($custom_fields as $tag => $value) {
        if ($value === NULL || $value === '' || (is_array($value) && empty($value))) {
          continue;
        }
        $processed_fields->{$tag} = $value;
      }
      if (!empty((array) $processed_fields)) {
        $data['custom_fields'] = $processed_fields;
      }
    }

    $url = self::API_BASE_URL . '/member';
    $options = [
      'headers' => [
        'Authorization' => $this->getAuthHeader(),
        'Content-Type' => 'application/json',
      ],
      'json' => $data,
      'http_errors' => FALSE,
    ];

    try {
      $response = $this->httpClient->request('POST', $url, $options);
      $status_code = $response->getStatusCode();
      $body = json_decode($response->getBody(), TRUE);

      if ($status_code === 201) {
        return [
          'success' => TRUE,
          'message' => 'Successfully subscribed.',
        ];
      }

      if ($status_code === 204) {
        return [
          'success' => TRUE,
          'message' => 'Email address already subscribed.',
        ];
      }

      $error_message = $body['error']['message'] ?? 'Unknown error';
      $error_code = $body['error']['code'] ?? $status_code;

      $this->logger->warning('Laposta subscription failed: @error (Code: @code)', [
        '@error' => $error_message,
        '@code' => $error_code,
      ]);

      return [
        'success' => FALSE,
        'error' => $error_message,
        'code' => $error_code,
      ];
    }
    catch (RequestException $e) {
      $this->logger->error('Laposta subscription request failed: @error', [
        '@error' => $e->getMessage(),
      ]);

      return [
        'success' => FALSE,
        'error' => 'Connection error: ' . $e->getMessage(),
      ];
    }
  }

}
