<?php

namespace Drupal\laposta_webform\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Drupal\laposta_webform\Service\LapostaApi;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the Laposta Webform help page.
 */
class LapostaWebformHelpController extends ControllerBase {

  /**
   * The Laposta API service.
   *
   * @var \Drupal\laposta_webform\Service\LapostaApi
   */
  protected $lapostaApi;

  /**
   * Constructs a LapostaWebformHelpController object.
   *
   * @param \Drupal\laposta_webform\Service\LapostaApi $laposta_api
   *   The Laposta API service.
   */
  public function __construct(LapostaApi $laposta_api) {
    $this->lapostaApi = $laposta_api;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('laposta_webform.api')
    );
  }

  /**
   * Displays the help page.
   *
   * @return array
   *   A render array for the help page.
   */
  public function help() {
    $help_email = 'helpdesk@laposta.nl';
    $author_email = 'fred@miix.dev';

    // Get API info.
    $api_info = $this->lapostaApi->getApiInfo();

    $build = [
      '#theme' => 'laposta_webform_help',
      '#content' => [
        'about' => [
          'title' => $this->t('About'),
          'content' => $this->t('The Laposta Webform module integrates Drupal Webform with Laposta mailing lists, allowing you to subscribe form submissions to one or more lists with custom field mapping.'),
        ],
        'getting_started' => [
          'title' => $this->t('Getting Started'),
          'items' => [
            $this->t('Configure your Laposta API key at <a href="@url">Configuration > Web services > Laposta Webform</a>.', ['@url' => Url::fromRoute('laposta_webform.settings')->toString()]),
            $this->t('Edit your webform and go to Settings > Emails / Handlers.'),
            $this->t('Click "Add handler" and select "Laposta".'),
            $this->t('Configure the handler settings (list, email field, field mapping).'),
          ],
        ],
        'handler_features' => [
          'title' => $this->t('Handler Features'),
          'items' => [
            $this->t('<strong>Fixed list subscription:</strong> Subscribe all submissions to a specific list.'),
            $this->t('<strong>Dynamic list selection:</strong> Let visitors choose their list(s) via form elements.'),
            $this->t('<strong>Multiple list support:</strong> Subscribe to multiple lists at once.'),
            $this->t('<strong>Opt-in field:</strong> GDPR-compliant checkbox before subscribing.'),
            $this->t('<strong>Field mapping:</strong> Map webform fields to Laposta custom fields.'),
            $this->t('<strong>Required field validation:</strong> Validates before subscription.'),
          ],
        ],
        'list_elements' => [
          'title' => $this->t('Laposta List Elements'),
          'items' => [
            $this->t('<strong>Laposta list select:</strong> Dropdown populated with your mailing lists.'),
            $this->t('<strong>Laposta list checkboxes:</strong> Checkboxes for multiple list selection.'),
            $this->t('<strong>List filtering:</strong> Restrict which lists visitors can see.'),
          ],
        ],
        'field_types' => [
          'title' => $this->t('Supported Field Types'),
          'content' => $this->t('Text, Numeric, Date, Single select, and Multiple select.'),
        ],
        'api_status' => [
          'title' => $this->t('API Connection Status'),
          'status' => $api_info['status'],
          'message' => $api_info['message'],
          'account_name' => $api_info['account_name'] ?? $this->t('Not available'),
          'account_number' => $api_info['account_number'] ?? $this->t('Not available'),
        ],
        'further_assistance' => [
          'title' => $this->t('Need Further Assistance?'),
          'content' => $this->t('For Laposta-related questions, contact the Laposta support team at:'),
          'email' => $help_email,
        ],
        'about_author' => [
          'title' => $this->t('About the Author'),
          'content' => $this->t('This module was created by Fred Klopper. For module-specific inquiries or support, you can contact the author at:'),
          'email' => $author_email,
        ],
      ],
    ];

    $build['#attached']['library'][] = 'laposta_webform/help';

    return $build;
  }

}
